"use strict";

/**
 * Enhanced Dashboard JavaScript
 * Features: Interactive charts, legends, tooltips, animations
 */

(function() {
    // Chart instance reference
    let chartInstance = null;
    let currentRange = '30d';
    let fullData = null;

    /**
     * Initialize the dashboard
     */
    function init() {
        initializeChart();
        initializeLegendToggles();
        initializeRangeButtons();
        initializeCopyButtons();
        initializeAnimations();
        initializeImageGallery();
    }

    /**
     * Get chart data from DOM data attributes
     */
    function getChartData() {
        const dataEl = document.getElementById('dashboard-chart-data');
        if (!dataEl) {
            console.warn('Dashboard chart data element not found');
            return null;
        }

        try {
            return {
                dates: JSON.parse(dataEl.dataset.dates || '[]'),
                documents: JSON.parse(dataEl.dataset.documents || '[]'),
                images: JSON.parse(dataEl.dataset.images || '[]'),
                codes: JSON.parse(dataEl.dataset.codes || '[]'),
                videos: JSON.parse(dataEl.dataset.videos || '[]'),
                chats: JSON.parse(dataEl.dataset.chats || '[]')
            };
        } catch (e) {
            console.error('Error parsing chart data:', e);
            return null;
        }
    }

    /**
     * Initialize the usage statistics chart
     */
    function initializeChart() {
        const canvas = document.getElementById('myChart');
        if (!canvas) return;

        // Get data from DOM
        fullData = getChartData();
        if (!fullData) {
            // Fallback to global variables if data attributes not available
            if (typeof dates !== 'undefined') {
                fullData = {
                    dates: dates || [],
                    documents: typeof contents !== 'undefined' ? contents : [],
                    images: typeof images !== 'undefined' ? images : [],
                    codes: typeof codes !== 'undefined' ? codes : [],
                    videos: typeof videos !== 'undefined' ? videos : (typeof codes !== 'undefined' ? codes : []),
                    chats: typeof chats !== 'undefined' ? chats : []
                };
            } else {
                console.warn('No chart data available');
                return;
            }
        }

        const ctx = canvas.getContext('2d');
        const isDarkMode = document.documentElement.classList.contains('dark');
        
        // Color palette with enhanced gradients
        const colors = {
            documents: { border: '#763CD4', bg: 'rgba(118, 60, 212, 0.15)', bgHover: 'rgba(118, 60, 212, 0.25)' },
            images: { border: '#E22861', bg: 'rgba(226, 40, 97, 0.15)', bgHover: 'rgba(226, 40, 97, 0.25)' },
            videos: { border: '#3B82F6', bg: 'rgba(59, 130, 246, 0.15)', bgHover: 'rgba(59, 130, 246, 0.25)' },
            codes: { border: '#FF774B', bg: 'rgba(255, 119, 75, 0.15)', bgHover: 'rgba(255, 119, 75, 0.25)' },
            chats: { border: '#10B981', bg: 'rgba(16, 185, 129, 0.15)', bgHover: 'rgba(16, 185, 129, 0.25)' }
        };

        // Create gradient fills for each dataset
        const createGradient = (ctx, color) => {
            const gradient = ctx.createLinearGradient(0, 0, 0, 260);
            gradient.addColorStop(0, color.bgHover);
            gradient.addColorStop(1, 'rgba(255, 255, 255, 0)');
            return gradient;
        };

        const gridColor = isDarkMode ? 'rgba(255, 255, 255, 0.06)' : 'rgba(0, 0, 0, 0.06)';
        const textColor = isDarkMode ? '#9CA3AF' : '#6B7280';

        // Custom legend plugin
        const legendPlugin = {
            id: 'legendMargin',
            beforeInit(chart) {
                const fitValue = chart.legend.fit;
                chart.legend.fit = function fit() {
                    fitValue.bind(chart.legend)();
                    return this.height += 16;
                };
            }
        };

        const config = {
            type: 'line',
            data: {
                labels: fullData.dates,
                datasets: [
                    {
                        label: 'Documents',
                        data: fullData.documents,
                        borderColor: colors.documents.border,
                        backgroundColor: createGradient(ctx, colors.documents),
                        borderWidth: 2.5,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 7,
                        pointBackgroundColor: colors.documents.border,
                        pointBorderColor: isDarkMode ? '#292929' : '#fff',
                        pointBorderWidth: 3,
                        pointHoverBorderWidth: 4
                    },
                    {
                        label: 'Images',
                        data: fullData.images,
                        borderColor: colors.images.border,
                        backgroundColor: createGradient(ctx, colors.images),
                        borderWidth: 2.5,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 7,
                        pointBackgroundColor: colors.images.border,
                        pointBorderColor: isDarkMode ? '#292929' : '#fff',
                        pointBorderWidth: 3,
                        pointHoverBorderWidth: 4
                    },
                    {
                        label: 'Videos',
                        data: fullData.videos,
                        borderColor: colors.videos.border,
                        backgroundColor: createGradient(ctx, colors.videos),
                        borderWidth: 2.5,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 7,
                        pointBackgroundColor: colors.videos.border,
                        pointBorderColor: isDarkMode ? '#292929' : '#fff',
                        pointBorderWidth: 3,
                        pointHoverBorderWidth: 4
                    },
                    {
                        label: 'Codes',
                        data: fullData.codes,
                        borderColor: colors.codes.border,
                        backgroundColor: createGradient(ctx, colors.codes),
                        borderWidth: 2.5,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 7,
                        pointBackgroundColor: colors.codes.border,
                        pointBorderColor: isDarkMode ? '#292929' : '#fff',
                        pointBorderWidth: 3,
                        pointHoverBorderWidth: 4
                    },
                    {
                        label: 'Chats',
                        data: fullData.chats,
                        borderColor: colors.chats.border,
                        backgroundColor: createGradient(ctx, colors.chats),
                        borderWidth: 2.5,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 7,
                        pointBackgroundColor: colors.chats.border,
                        pointBorderColor: isDarkMode ? '#292929' : '#fff',
                        pointBorderWidth: 3,
                        pointHoverBorderWidth: 4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false
                },
                plugins: {
                    legend: {
                        display: false // We use custom legend
                    },
                    tooltip: {
                        enabled: true,
                        backgroundColor: isDarkMode ? '#1F2937' : '#ffffff',
                        titleColor: isDarkMode ? '#ffffff' : '#111827',
                        bodyColor: isDarkMode ? '#D1D5DB' : '#4B5563',
                        borderColor: isDarkMode ? '#374151' : '#E5E7EB',
                        borderWidth: 1,
                        padding: 14,
                        cornerRadius: 10,
                        displayColors: true,
                        usePointStyle: true,
                        boxPadding: 6,
                        titleFont: {
                            size: 13,
                            weight: 'bold',
                            family: 'Figtree, sans-serif'
                        },
                        bodyFont: {
                            size: 12,
                            family: 'Figtree, sans-serif'
                        },
                        callbacks: {
                            title: function(context) {
                                return '📅 ' + context[0].label;
                            },
                            label: function(context) {
                                return '  ' + context.dataset.label + ': ' + context.parsed.y;
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: textColor,
                            font: {
                                size: 11,
                                family: 'Figtree, sans-serif'
                            },
                            maxRotation: 0
                        }
                    },
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: gridColor,
                            drawBorder: false
                        },
                        ticks: {
                            color: textColor,
                            font: {
                                size: 11,
                                family: 'Figtree, sans-serif'
                            },
                            stepSize: 1,
                            callback: function(value) {
                                return Number.isInteger(value) ? value : '';
                            }
                        }
                    }
                },
                animation: {
                    duration: 750,
                    easing: 'easeInOutQuart'
                }
            },
            plugins: [legendPlugin]
        };

        // Destroy existing instance
        if (chartInstance) {
            chartInstance.destroy();
        }

        chartInstance = new Chart(ctx, config);
    }

    /**
     * Initialize custom legend toggle functionality
     */
    function initializeLegendToggles() {
        // Support both old and new legend classes
        const legendItems = document.querySelectorAll('.chart-legend-item, .chart-legend-btn');
        
        legendItems.forEach(item => {
            item.addEventListener('click', function() {
                if (!chartInstance) return;
                
                const datasetLabel = this.dataset.dataset;
                const datasetIndex = chartInstance.data.datasets.findIndex(
                    ds => ds.label === datasetLabel
                );
                
                if (datasetIndex === -1) return;
                
                const meta = chartInstance.getDatasetMeta(datasetIndex);
                meta.hidden = meta.hidden === null ? !chartInstance.data.datasets[datasetIndex].hidden : null;
                
                // Update legend item visual state (support both class naming conventions)
                const isHidden = meta.hidden === true;
                this.classList.toggle('disabled', isHidden);
                this.classList.toggle('active', !isHidden);
                
                chartInstance.update('none');
            });
        });
    }

    /**
     * Initialize chart range buttons (7D, 30D, All)
     */
    function initializeRangeButtons() {
        const rangeButtons = document.querySelectorAll('.chart-range-btn');
        
        rangeButtons.forEach(btn => {
            btn.addEventListener('click', function() {
                const range = this.dataset.range;
                if (range === currentRange) return;
                
                // Update active state
                rangeButtons.forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                
                currentRange = range;
                updateChartRange(range);
            });
        });
    }

    /**
     * Update chart data based on selected range
     * 7D = last 7 days from today (dynamic)
     * 30D = last 30 days from today (dynamic)
     * All = all available data
     */
    function updateChartRange(range) {
        if (!chartInstance || !fullData) return;
        
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        
        let filteredIndices = [];
        
        if (range === 'all') {
            // All data
            filteredIndices = fullData.dates.map((_, idx) => idx);
        } else {
            // Calculate days to go back
            const daysBack = range === '7d' ? 7 : 30;
            const cutoffDate = new Date(today);
            cutoffDate.setDate(cutoffDate.getDate() - daysBack + 1);
            
            // Filter dates that fall within the range
            fullData.dates.forEach((dateStr, idx) => {
                // Parse the date string (assuming format like "1", "2", etc. for days of month)
                // or full date strings
                const dayNum = parseInt(dateStr);
                if (!isNaN(dayNum)) {
                    // If it's just a day number, check if it's within the last N days
                    const currentMonth = today.getMonth();
                    const currentYear = today.getFullYear();
                    const checkDate = new Date(currentYear, currentMonth, dayNum);
                    
                    // If the day is in the future this month, it's from last month
                    if (checkDate > today) {
                        checkDate.setMonth(checkDate.getMonth() - 1);
                    }
                    
                    if (checkDate >= cutoffDate && checkDate <= today) {
                        filteredIndices.push(idx);
                    }
                } else {
                    // If we can't parse, just use the last N entries
                    filteredIndices.push(idx);
                }
            });
            
            // If no indices matched, fall back to last N entries
            if (filteredIndices.length === 0) {
                const startIdx = Math.max(0, fullData.dates.length - (range === '7d' ? 7 : 30));
                filteredIndices = Array.from({length: fullData.dates.length - startIdx}, (_, i) => startIdx + i);
            }
        }
        
        // Extract data for filtered indices
        const labels = filteredIndices.map(i => fullData.dates[i]);
        const documents = filteredIndices.map(i => fullData.documents[i] || 0);
        const images = filteredIndices.map(i => fullData.images[i] || 0);
        const videos = filteredIndices.map(i => fullData.videos[i] || 0);
        const codes = filteredIndices.map(i => fullData.codes[i] || 0);
        const chats = filteredIndices.map(i => fullData.chats[i] || 0);
        
        chartInstance.data.labels = labels;
        chartInstance.data.datasets[0].data = documents;
        chartInstance.data.datasets[1].data = images;
        chartInstance.data.datasets[2].data = videos;
        chartInstance.data.datasets[3].data = codes;
        chartInstance.data.datasets[4].data = chats;
        
        chartInstance.update('active');
    }

    /**
     * Initialize coupon copy functionality
     */
    function initializeCopyButtons() {
        const copyButtons = document.querySelectorAll('.copy-button');
        
        copyButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                // Try to find coupon code in different parent structures
                let couponCodeEl = null;
                let messageEl = this.querySelector('.copied-message');
                
                // Try new card structure first
                const couponCard = this.closest('.coupon-card-enhanced');
                if (couponCard) {
                    couponCodeEl = couponCard.querySelector('[data-coupon-code]');
                    if (!messageEl) {
                        messageEl = this.querySelector('.copied-message');
                    }
                }
                
                // Fallback to old structure
                if (!couponCodeEl) {
                    const magicBg = this.closest('.magic-bg');
                    if (magicBg) {
                        couponCodeEl = magicBg.querySelector('[data-coupon-code]');
                        messageEl = magicBg.querySelector('.copied-message');
                    }
                }
                
                // Also try coupon-code-box parent
                if (!couponCodeEl) {
                    const codeBox = this.closest('.coupon-code-box');
                    if (codeBox) {
                        couponCodeEl = codeBox.querySelector('[data-coupon-code]');
                    }
                }
                
                if (!couponCodeEl) return;
                
                const couponCode = couponCodeEl.textContent.trim();
                
                navigator.clipboard.writeText(couponCode).then(() => {
                    if (messageEl) {
                        messageEl.classList.remove('hidden');
                        messageEl.style.display = 'block';
                        setTimeout(() => {
                            messageEl.classList.add('hidden');
                            messageEl.style.display = 'none';
                        }, 1500);
                    }
                }).catch((err) => {
                    console.error('Failed to copy:', err);
                    fallbackCopy(couponCode, messageEl);
                });
            });
        });
    }

    /**
     * Fallback copy method for older browsers
     */
    function fallbackCopy(text, messageEl) {
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.left = '-9999px';
        document.body.appendChild(textArea);
        textArea.select();
        
        try {
            document.execCommand('copy');
            if (messageEl) {
                messageEl.textContent = 'Copied!';
                messageEl.style.display = 'block';
                setTimeout(function() {
                    messageEl.style.display = 'none';
                }, 1500);
            }
        } catch (err) {
            console.error('Fallback copy failed:', err);
        }
        
        document.body.removeChild(textArea);
    }

    /**
     * Initialize scroll-triggered animations
     */
    function initializeAnimations() {
        const animatedElements = document.querySelectorAll('.animate-on-scroll');
        
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-fade-in');
                        observer.unobserve(entry.target);
                    }
                });
            }, {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            });
            
            animatedElements.forEach(el => observer.observe(el));
        } else {
            // Fallback: just show elements
            animatedElements.forEach(el => el.classList.add('animate-fade-in'));
        }
    }

    /**
     * Handle theme changes - reinitialize chart with new colors
     */
    function handleThemeChange() {
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.attributeName === 'class') {
                    // Debounce chart reinitialization
                    clearTimeout(window.chartThemeTimeout);
                    window.chartThemeTimeout = setTimeout(() => {
                        initializeChart();
                    }, 300);
                }
            });
        });

        observer.observe(document.documentElement, {
            attributes: true,
            attributeFilter: ['class']
        });
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            init();
            handleThemeChange();
        });
    } else {
        init();
        handleThemeChange();
    }

    /**
     * Initialize enhanced image gallery interactions
     */
    function initializeImageGallery() {
        const thumbnailCards = document.querySelectorAll('.image-thumbnail-card');

        thumbnailCards.forEach((card, index) => {
            // Add staggered animation on load
            card.style.animationDelay = `${index * 50}ms`;
            card.classList.add('animate-fade-in');

            // Enhanced hover interactions
            card.addEventListener('mouseenter', function() {
                // Add subtle vibration effect
                this.style.animation = 'none';
                setTimeout(() => {
                    this.style.animation = 'pulse 0.6s ease-in-out';
                }, 10);

                // Enhance sibling cards (dim them slightly)
                const siblings = Array.from(thumbnailCards).filter(c => c !== this);
                siblings.forEach(sibling => {
                    sibling.style.opacity = '0.7';
                    sibling.style.transform = 'scale(0.98)';
                });
            });

            card.addEventListener('mouseleave', function() {
                this.style.animation = '';
                this.style.transform = '';

                // Reset sibling cards
                thumbnailCards.forEach(sibling => {
                    sibling.style.opacity = '';
                    sibling.style.transform = '';
                });
            });

            // Add click tracking for analytics
            card.addEventListener('click', function() {
                // Simple click tracking - could be enhanced with analytics
                console.log('Image thumbnail clicked:', this.querySelector('img').alt);
            });
        });

        // Enhanced featured image interactions
        const featuredImage = document.querySelector('.image-banner-featured img');
        if (featuredImage) {
            featuredImage.addEventListener('load', function() {
                // Add subtle parallax effect on mouse move
                const container = this.closest('.image-banner-featured');
                container.addEventListener('mousemove', function(e) {
                    const rect = container.getBoundingClientRect();
                    const x = (e.clientX - rect.left) / rect.width;
                    const y = (e.clientY - rect.top) / rect.height;

                    const moveX = (x - 0.5) * 10;
                    const moveY = (y - 0.5) * 10;

                    featuredImage.style.transform = `scale(1.05) translate(${moveX}px, ${moveY}px)`;
                });

                container.addEventListener('mouseleave', function() {
                    featuredImage.style.transform = '';
                });
            });
        }

        // Action button enhancements
        const actionButtons = document.querySelectorAll('.image-banner-action-btn');
        actionButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                // Add ripple effect
                const ripple = document.createElement('div');
                ripple.style.cssText = `
                    position: absolute;
                    border-radius: 50%;
                    background: rgba(255, 255, 255, 0.6);
                    transform: scale(0);
                    animation: ripple 0.6s linear;
                    width: 20px;
                    height: 20px;
                    left: 50%;
                    top: 50%;
                    margin-left: -10px;
                    margin-top: -10px;
                `;

                this.appendChild(ripple);
                setTimeout(() => ripple.remove(), 600);
            });
        });
    }

    // Add ripple animation keyframes if not already present
    if (!document.querySelector('#ripple-keyframes')) {
        const style = document.createElement('style');
        style.id = 'ripple-keyframes';
        style.textContent = `
            @keyframes ripple {
                to {
                    transform: scale(4);
                    opacity: 0;
                }
            }
        `;
        document.head.appendChild(style);
    }

    // Export for external use
    window.DashboardEnhanced = {
        reinitChart: initializeChart,
        updateRange: updateChartRange,
        initImageGallery: initializeImageGallery
    };

})();

/**
 * Global Download Image Function
 * Handles image downloads - simple and direct approach
 */
window.downloadImage = function(imageUrl, imageName, buttonElement) {
    // Prevent event propagation
    if (window.event) {
        window.event.preventDefault();
        window.event.stopPropagation();
        window.event.stopImmediatePropagation();
    }
    
    // Get the button element
    const button = buttonElement || (window.event && window.event.currentTarget);
    
    if (button) {
        const originalHTML = button.innerHTML;
        
        // Show loading indicator
        button.innerHTML = `
            <svg class="w-4 h-4 animate-spin" fill="none" viewBox="0 0 24 24">
                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
            </svg>
        `;
        button.disabled = true;
        
        // Use simple approach - create hidden link and click it
        setTimeout(() => {
            const link = document.createElement('a');
            link.href = imageUrl;
            link.download = imageName || 'image.png';
            link.style.display = 'none';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            // Show success
            button.innerHTML = `
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                </svg>
            `;
            
            // Reset button
            setTimeout(() => {
                button.innerHTML = originalHTML;
                button.disabled = false;
            }, 1500);
        }, 100);
    } else {
        // No button, just download directly
        const link = document.createElement('a');
        link.href = imageUrl;
        link.download = imageName || 'image.png';
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }
    
    return false; // Prevent default action
};

