<?php

/**
 * @package ContentService
 * @author TechVillage <support@techvill.org>
 * @contributor Kabir Ahmed <[kabir.techvill@gmail.com]>
 * @created 07-03-2023
 */

namespace Modules\OpenAI\Services;

use App\Models\{
    Language,
    User,
    Team,
    TeamMemberMeta
};
use App\Traits\ApiResponse;
use Modules\OpenAI\Entities\{
    UseCase,
    Archive
};
use Modules\OpenAI\Traits\MetaTrait;

 class ContentService
 {
    use MetaTrait, ApiResponse;
    protected $formData;
    protected $preparedData;
    protected $response;
    protected $failedResponse;

    /**
     * Initialize
     *
     * @param string $service
     * @return void
     */
    public function __construct($formData = null, $preparedData = null, $response = null, $failedResponse = null)
    {
        $this->formData = $formData;
        $this->preparedData = $preparedData;
        $this->response = $response;
        $this->failedResponse = [
            'status' => 'failed',
        ];
    }

    /**
     * Team member meta insert or update
     * @param array $imagedata
     *
     * @return bool|array
     */
    public function storeTeamMeta($words)
    {
        $memberData = Team::getMember(auth()->user()->id);
        if (!empty($memberData)) {
            $usage = TeamMemberMeta::getMemberMeta($memberData->id, 'word_used');
            if (!empty($usage)) {
                return $usage && $usage->increment('value', $words); 
            }
        }
        return false;
    }

    /**
     * Get All Data
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAll()
    {
        $result = $this->model()->with('useCase');
        $result = $result->where('user_id', auth()->user()->id);

        return $result->whereNull('parent_id')->latest();
    }

    /**
     * Get All Favorite
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAllFavourite()
    {
        $bookmarks = auth()->user()->document_bookmarks_openai;

        $result = $this->model()->with(['useCase'])->whereIn('id', $bookmarks);
        
        return $result->where('type', 'template_chat')->latest();
    }

    /**
     * Use Cases
     *
     * @param array $data
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function useCases($data = null)
    {
        $useCases = UseCase::where('status', 'Active')->get();

        if ($data != null) {

            $favUseCases = $useCases->whereIn('id', $data);
            $exceptFavUseCases = $useCases->whereNotIn('id', $data);

            $useCases =  $favUseCases->merge($exceptFavUseCases);

        }

        return $useCases;

    }

    /**
     * Language Data
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function languages()
    {
        return Language::where(['status' => 'Active'])->get();
    }

    /**
     * Users Data
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function users()
    {
        return User::where(['status' => 'Active'])->get();
    }

    /**
     * Content Model
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public static function model()
    {
        return Archive::with(['metas']);
    }

    /**
     * Get current member from session or meta
     * 
     * @param string $currentPackageMeta
     * @param string $currentPackageSession
     * @return [type]
     */
    
    /**
     * Get the effective user ID for subscription and billing purposes
     *
     * For team members, returns the team owner's ID
     * For regular users, returns their own ID
     *
     * @param string|null $mode 'meta' for package data, 'session' for session data
     * @param array $data Optional data array with 'user_id' key
     * @return int|array Returns user ID or error response array
     */
    public function getCurrentMemberUserId(?string $mode = null, ?string $sessionMode = null, array $data = [])
    {
        $userId = $data['user_id'] ?? auth()->id();

        // Check if user is a team member
        $memberData = Team::getMember($userId);

        if ($memberData) {
            return $this->handleTeamMemberUserId($memberData, $userId, $mode, $sessionMode);
        }

        // Not a team member, return their own ID
        return $userId;
    }

    /**
     * Handle user ID resolution for team members
     *
     * @param object $memberData Team member data
     * @param int $userId Current user ID
     * @param string|null $mode Operation mode
     * @param string|null $sessionMode Session mode
     * @return int|array
     */
    private function handleTeamMemberUserId($memberData, int $userId, ?string $mode, ?string $sessionMode)
    {
        // If user doesn't have credit details, return team owner's ID
        if (!subscription('hasCreditDetails', $userId)) {
            return $memberData->user_id; // Team owner ID
        }

        // Handle different modes
        if ($mode === 'meta') {
            return $this->getPackageMetaUserId();
        }

        if ($sessionMode === 'session') {
            return $this->getSessionUserId();
        }

        return $this->unprocessableResponse([
            'response' => __('Invalid mode specified'),
            'status' => 'failed',
        ]);
    }

    /**
     * Get user ID from package meta data
     *
     * @return int|array
     */
    private function getPackageMetaUserId()
    {
        $memberPackageData = Team::memberSession();

        if ($memberPackageData) {
            return $memberPackageData->value;
        }

        return $this->unprocessableResponse([
            'response' => __('Please set your plan from settings'),
            'status' => 'failed',
        ]);
    }

    /**
     * Get user ID from session data
     *
     * @return int
     */
    private function getSessionUserId(): int
    {
        return session('memberPackageData.packageUser', 0);
    }

    /**
     * Check parent or member user status
     * 
     * @param string $userId
     * @param string $type
     * @return [type]
     */
    public function checkUserStatus($userId, $type)
    {
        if ($type == 'meta') {
            $userStatus = User::where(['id' => $userId, 'status' => 'Active'])->first();
            if (!empty($userStatus) || $userId == 0) {
                return [
                    'message' => __('Subscribed user is active'),
                    'status' => 'success',
                ];
            } else {
                return [
                    'message' => __('The account of the user whose credits you\'re using is not active. Kindly reach out to administration for help.'),
                    'status' => 'fail',
                ];
            }
        }
    }

    /**
     * Retrieves all features.
     *
     * This method returns a collection of features, including their names, descriptions, access levels, and icons.
     *
     * @return array An array of features with their names, descriptions, access levels, and icons.
     */
    public function allFeatures(array $data = [])
    {
        $slug = $data['slug'] ?? 'demo';
        $id = $data['id'] ?? 'demo';
        $color1 = '#141414';
        $color2 = '#141414';

        // Category order matching Blade sidebar
        $categoryOrder = [
            'Content Creation',
            'Content Analysis',
            'Visual Studio',
            'Video Studio',
            'Voice & Audio',
            'AI Conversational Tools',
            'Marketing Tools',
            'AI Influencer',
            'System Management'
        ];

        // Base SVG icons (raw SVG elements)
        $baseCategoryIcons = [
            'Content Creation' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/></svg>',
            'Content Analysis' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/></svg>',
            'Visual Studio' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/></svg>',
            'Video Studio' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"/></svg>',
            'Voice & Audio' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z"/></svg>',
            'AI Conversational Tools' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z"/></svg>',
            'Marketing Tools' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5.882V19.24a1.76 1.76 0 01-3.417.592l-2.147-6.15M18 13a3 3 0 100-6M5.436 13.683A4.001 4.001 0 017 6h1.832c4.1 0 7.625-1.234 9.168-3v14c-1.543-1.766-5.067-3-9.168-3H7a3.988 3.988 0 01-1.564-.317z"/></svg>',
            'AI Influencer' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5.121 17.804A13.937 13.937 0 0112 16c2.5 0 4.847.655 6.879 1.804M15 10a3 3 0 11-6 0 3 3 0 016 0zm6 2a9 9 0 11-18 0 9 9 0 0118 0z"/></svg>',
            'System Management' => '<svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/></svg>',
        ];

        // Category icons as SVG strings (Blade compatible) - wrapped in span elements
        $categoryIcons = array_map(function($svg) {
            return '<span class="h-5 w-5 category-svg text-color-14 dark:text-white transition-colors duration-200">' . $svg . '</span>';
        }, $baseCategoryIcons);

        // Category icons as React components (for PrimarySideNav.jsx) - raw SVG elements
        $categoryIconsReact = $baseCategoryIcons;

        // Build categories array
        $categories = [];
        foreach ($categoryOrder as $index => $categoryName) {
            $categories[] = [
                'name' => $categoryName,
                'icon' => $categoryIcons[$categoryName] ?? '',
                'icon_react' => $categoryIconsReact[$categoryName] ?? '',
                'order' => $index,
            ];
        }

        $dashboard = [
            'id' => 'dashboard',
            'name' => __('Dashboard'),
            'description' => __('View your dashboard and manage your content.'),
            'icon' => '<span class="h-5 w-5 category-svg">
                            <svg width="16" height="17" viewBox="0 0 16 17" fill="none"
                            xmlns="http://www.w3.org/2000/svg">
                            <path
                                d="M5.5 13.1664H10.5V16.4998H5.5V13.1664ZM0.5 15.2164L3.83333 11.8831V16.4998H1.33333C0.875 16.4998 0.5 16.1248 0.5 15.6664V15.2164ZM10.3583 2.99976L3.83333 9.52476L0.5 12.8498V8.49143C0.5 8.28309 0.575 8.09143 0.708333 7.94143C0.716667 7.92476 0.733333 7.91643 0.741667 7.89976L7.40833 1.23309C7.43333 1.20809 7.45 1.19143 7.475 1.18309L7.53333 1.12476C7.86667 0.916428 8.30833 0.949761 8.59167 1.23309L10.3583 2.99976ZM15.5 8.49143V15.6664C15.5 16.1248 15.125 16.4998 14.6667 16.4998H12.1667V9.52476L9.18333 6.54143L11.5417 4.18309L15.2583 7.89976C15.4083 8.05809 15.5 8.26643 15.5 8.49143Z"
                                fill="url(#paint0_linear_2671_2007)" />
                            <defs>
                                <linearGradient id="paint0_linear_2671_2007" x1="10.2526" y1="14.5909" x2="4.74869"
                                    y2="2.57816" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.dashboard'))['color1'] ?? $color1) . '" />
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.dashboard'))['color2'] ?? $color2) . '" />
                                </linearGradient>
                            </defs>
                        </svg>
                    </span>',
            'access' => true,
            'route' => route('user.dashboard'),
            'menu' => activeMenu(route('user.dashboard')),
            'type' => 'dashboard',
        ];

        $features = [
            [
                'id' => 'template',
                'category' => 'Content Creation',
                'name' => __('Pre-built Templates'),
                'icon' => '<span class="w-5 h-5 text-color-14 dark:text-white">
                                <svg class="category-svg" width="16" height="16" viewBox="0 0 16 16" fill="none"
                                xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M0.5 8.83333H7.16667V0.5H0.5V8.83333ZM0.5 15.5H7.16667V10.5H0.5V15.5ZM8.83333 15.5H15.5V7.16667H8.83333V15.5ZM8.83333 0.5V5.5H15.5V0.5H8.83333Z"
                                    fill="currentColor" />
                                <defs>
                                    <linearGradient id="paint0_linear_templates" x1="10.2526" y1="13.6538"
                                        x2="5.04573" y2="1.90371" gradientUnits="userSpaceOnUse">
                                        <stop stop-color="' . (activeMenu(route('openai'), route('user.template', ['slug' => $slug]))['color1'] ?? $color1) . '" />
                                        <stop offset="1" stop-color="' . (activeMenu(route('openai'), route('user.template', ['slug' => $slug]))['color2'] ?? $color2) . '" />
                                    </linearGradient>
                                </defs>
                            </svg>
                            </span>',
                'access' => hasAccess('template') && customerPanelAccess('template'),
                'route' =>  route('openai'),
                'menu' => activeMenu(route('openai'), route('user.template', ['slug' => $slug])),
                'settings' => [
                    'toggle_id' => 'template',
                    'label' => __('Template'),
                    'description' => __('Toggle to enable or disable the template features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'long_article',
                'category' => 'Content Creation',
                'name' => __('Long Article'),
                'description' => __('Generate comprehensive, well-structured long-form content and articles.'),
                'icon' => '<span class="w-4 h-4 text-color-14 dark:text-white">
                                <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M16.181 2.58949L17.4135 3.82165C18.1955 4.60347 18.1955 5.87002 17.4135 6.65184L15.7837 8.28116L15.7524 8.24989L15.2519 7.74952L12.2489 4.74732L11.7171 4.21568L13.3469 2.58637C14.1289 1.80454 15.3958 1.80454 16.1779 2.58637L16.181 2.58949ZM9.53996 5.57918C9.24592 5.28522 8.77044 5.28522 8.47952 5.57918L5.2857 8.77214C4.99166 9.06611 4.51618 9.06611 4.22526 8.77214C3.93435 8.47818 3.93122 8.00283 4.22526 7.71199L7.41596 4.51903C8.29496 3.64026 9.72139 3.64026 10.6004 4.51903L11.0102 4.92871L11.542 5.46035L14.545 8.46254L15.0455 8.96291L15.0768 8.99418L14.545 9.52582L9.18023 14.886C7.67872 16.3871 5.7643 17.4128 3.68097 17.8288L2.89893 17.9851C2.65181 18.0352 2.39843 17.957 2.22013 17.7787C2.04182 17.6005 1.96675 17.3472 2.01367 17.1001L2.17008 16.3183C2.58612 14.2355 3.61215 12.3216 5.11365 10.8205L9.94975 5.98886L9.53996 5.57918Z" fill="url(#paint0_linear_10232_5981)"/>
                                    <defs>
                                    <linearGradient id="paint0_linear_10232_5981" x1="12.4027" y1="16.0307" x2="6.84878" y2="3.49729" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.long_article.create'))['color1'] ?? $color1) . '"/>
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.long_article.create'))['color2'] ?? $color2) . '"/>
                                    </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' => hasAccess('long_article') && customerPanelAccess('long_article'),
                'route' => route('user.long_article.create'),
                'menu' => activeMenu(route('user.long_article.create')),
                'settings' => [
                    'toggle_id' => 'long_article',
                    'label' => __('Long Article'),
                    'description' => __('Toggle to enable or disable support for long articles.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'image',
                'category' => 'Visual Studio',
                'name' => __('Image Maker'),
                'description' => __('Generate high-quality images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg class="category-svg" width="16" height="16" viewBox="0 0 16 16" fill="none"
                                xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M2 2C1.44772 2 1 2.44772 1 3V13C1 13.5523 1.44772 14 2 14H14C14.5523 14 15 13.5523 15 13V3C15 2.44772 14.5523 2 14 2H2ZM4.5 6.5C4.5 5.67157 5.17157 5 6 5C6.82843 5 7.5 5.67157 7.5 6.5C7.5 7.32843 6.82843 8 6 8C5.17157 8 4.5 7.32843 4.5 6.5ZM3 11L6 8L8 10L11 7L13 9V11H3Z"
                                    fill="url(#paint0_linear_image)" />
                                <defs>
                                    <linearGradient id="paint0_linear_image" x1="10.2526" y1="13.6538"
                                        x2="5.04573" y2="1.90371" gradientUnits="userSpaceOnUse">
                                        <stop stop-color="' . (activeMenu(route('user.image.create'))['color1'] ?? $color1) . '" />
                                        <stop offset="1" stop-color="' . (activeMenu(route('user.image.create'))['color2'] ?? $color2) . '" />
                                    </linearGradient>
                                </defs>
                            </svg>
                            </span>',
                'access' => hasAccess('image') && customerPanelAccess('image'),
                'route' => route('user.image.create'),
                'menu' => activeMenu(route('user.image.create')),
                'settings' => [
                    'toggle_id' => 'image',
                    'label' => __('Image'),
                    'description' => __('Toggle to enable or disable image features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'ai_video',
                'category' => 'Video Studio',
                'name' => __('Video Maker'),
                'description' => __('Create professional videos using artificial intelligence technology.'),
                'icon' => '<span>
                                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" fill="#000000" height="22px" width="22px" version="1.1" id="Capa_1" viewBox="0 0 58 58" xml:space="preserve">
                                    <defs>
                                      <linearGradient id="gradient1" x1="0%" y1="0%" x2="100%" y2="100%">
                                        <stop stop-color="' . activeMenu(route('user.image-to-video.videoTempalte'))['color1'] . '" />
                                        <stop offset="1" stop-color="' . activeMenu(route('user.image-to-video.videoTempalte'))['color2'] . '" />
                                      </linearGradient>
                                    </defs>
                                    <path d="M57,6H1C0.448,6,0,6.447,0,7v44c0,0.553,0.448,1,1,1h56c0.552,0,1-0.447,1-1V7C58,6.447,57.552,6,57,6z M10,50H2v-9h8V50z   M10,39H2v-9h8V39z M10,28H2v-9h8V28z M10,17H2V8h8V17z M36.537,29.844l-11,7C25.374,36.947,25.187,37,25,37  c-0.166,0-0.331-0.041-0.481-0.123C24.199,36.701,24,36.365,24,36V22c0-0.365,0.199-0.701,0.519-0.877  c0.32-0.175,0.71-0.162,1.019,0.033l11,7C36.825,28.34,37,28.658,37,29S36.825,29.66,36.537,29.844z M56,50h-8v-9h8V50z M56,39h-8  v-9h8V39z M56,28h-8v-9h8V28z M56,17h-8V8h8V17z" fill="url(#gradient1)"/>
                                  </svg>
                                  
                            </span>',
                'access' => hasAccess('video') && customerPanelAccess('video'),
                'route' => route('user.image-to-video.videoTempalte'),
                'menu' => activeMenu(route('user.image-to-video.videoTempalte')),
                'settings' => [
                    'toggle_id' => 'video',
                    'label' => __('AI Video'),
                    'description' => __('Toggle to enable or disable the AI Video feature.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'text_to_video',
                'category' => 'Video Studio',
                'name' => __('Text To Video'),
                'description' => __('Transform written content into engaging video presentations.'),
                'icon' => '<span class="w-5 h-5">
                                <svg width="23" height="23" viewBox="0 0 300 300" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M150 126C137.297 126 127 136.297 127 149V255C127 267.703 137.297 278 150 278H256C268.703 278 279 267.703 279 255V149C279 136.297 268.703 126 256 126H150ZM149 140C144.582 140 141 143.582 141 148V256C141 260.418 144.582 264 149 264H257C261.418 264 265 260.418 265 256V148C265 143.582 261.418 140 257 140H149Z" fill="url(#paint0_linear_170_771)"/>
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M167 249.062L249.458 201.76L167.265 154L167 249.062ZM183.047 221.444L217.517 201.671L183.158 181.706L183.047 221.444Z" fill="url(#paint1_linear_170_771)"/>
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M21 46C21 33.2975 31.2975 23 44 23H150C162.703 23 173 33.2975 173 46V116H159V45C159 40.5817 155.418 37 151 37H43C38.5817 37 35 40.5817 35 45V153C35 157.418 38.5817 161 43 161H114V175H44C31.2975 175 21 164.703 21 152V46Z" fill="url(#paint2_linear_170_771)"/>
                                    <line x1="51" y1="61" x2="144" y2="61" stroke="url(#paint3_linear_170_771)" stroke-width="16"/>
                                    <line x1="51" y1="97" x2="144" y2="97" stroke="url(#paint4_linear_170_771)" stroke-width="16"/>
                                    <line x1="51" y1="134" x2="113" y2="134" stroke="url(#paint5_linear_170_771)" stroke-width="16"/>
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M234 57L191 57L191 49L237 49C239.761 49 242 51.2386 242 54V100.635L260.627 82.0077C262.19 80.4456 264.722 80.4456 266.284 82.0077C267.846 83.5698 267.846 86.1025 266.284 87.6646L240.828 113.12C239.266 114.683 236.734 114.683 235.172 113.12L209.716 87.6646C208.154 86.1025 208.154 83.5698 209.716 82.0077C211.278 80.4456 213.81 80.4456 215.373 82.0077L234 100.635V57Z" fill="url(#paint6_linear_170_771)"/>
                                    <path fill-rule="evenodd" clip-rule="evenodd" d="M70.4561 248.292L113.456 248.292L113.456 256.292L67.4561 256.292C64.6946 256.292 62.4561 254.053 62.4561 251.292V204.657L43.8286 223.284C42.2665 224.846 39.7339 224.846 38.1718 223.284C36.6097 221.722 36.6097 219.19 38.1718 217.627L63.6276 192.172C65.1897 190.609 67.7224 190.609 69.2845 192.172L94.7403 217.627C96.3024 219.19 96.3024 221.722 94.7403 223.284C93.1782 224.846 90.6456 224.846 89.0835 223.284L70.4561 204.657V248.292Z" fill="url(#paint7_linear_170_771)"/>
                                    <defs>
                                    <linearGradient id="paint0_linear_170_771" x1="26" y1="25.5" x2="279" y2="278" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint1_linear_170_771" x1="26" y1="26" x2="283.5" y2="283" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint2_linear_170_771" x1="21" y1="23" x2="282" y2="276.5" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint3_linear_170_771" x1="97.5" y1="69" x2="97.5" y2="70" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint4_linear_170_771" x1="97.5" y1="105" x2="97.5" y2="106" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] . '"/>
                                    </linearGradient>
                                    <linearGradient id="paint5_linear_170_771" x1="82" y1="142" x2="82" y2="143" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint6_linear_170_771" x1="229.228" y1="49" x2="229.228" y2="114.292" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    <linearGradient id="paint7_linear_170_771" x1="37.0002" y1="223.646" x2="113.456" y2="223.646" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.text-to-video.template'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.text-to-video.template'))['color2'] .'"/>
                                    </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' => hasAccess('text_to_video') && customerPanelAccess('video'),
                'route' => route('user.text-to-video.template'),
                'menu' => activeMenu(route('user.text-to-video.template')),
                'settings' => [
                    'toggle_id' => 'text_to_video',
                    'label' => __('Text to Video'),
                    'description' => __('Toggle to enable or disable Text To Video features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'code',
                'category' => 'Content Creation',
                'name' => __('Code Writer'),
                'description' => __('Generate, debug, and optimize code across multiple programming languages.'),
                'icon' => '<span class="w-5 h-5">
                                <svg class="category-svg" width="20" height="16" viewBox="0 0 20 16" fill="none"
                                    xmlns="http://www.w3.org/2000/svg">
                                    <path
                                        d="M19.8761 7.69945L14.7717 2.59512C14.613 2.43639 14.3322 2.43639 14.1735 2.59512L13.0668 3.70174C12.9016 3.86702 12.9016 4.13495 13.0668 4.30023L16.7655 7.99849L13.067 11.6974C12.9018 11.8627 12.9018 12.1306 13.067 12.2959L14.1737 13.4025C14.253 13.4819 14.3607 13.5267 14.4729 13.5267C14.5849 13.5267 14.6928 13.4819 14.7722 13.4025L19.8761 8.29837C20.0414 8.13287 20.0414 7.86474 19.8761 7.69945Z"
                                        fill="url(#paint0_linear_2838_1839)" />
                                    <path
                                        d="M6.93299 11.6975L3.23494 7.99907L6.93362 4.30081C7.01298 4.22145 7.05764 4.11394 7.05764 4.00156C7.05764 3.8894 7.0132 3.78168 6.93362 3.70232L5.827 2.5957C5.74764 2.51633 5.63992 2.47168 5.52776 2.47168C5.41559 2.47168 5.30787 2.51633 5.22851 2.5957L0.123963 7.69961C-0.041321 7.86489 -0.041321 8.13282 0.123963 8.29831L5.2283 13.4024C5.30766 13.4818 5.41538 13.5267 5.52754 13.5267C5.63971 13.5267 5.74743 13.4818 5.82679 13.4024L6.93341 12.2958C7.09869 12.1307 7.09869 11.8628 6.93299 11.6975Z"
                                        fill="url(#paint1_linear_2838_1839)" />
                                    <path
                                        d="M12.9025 0.877949C12.8488 0.779328 12.7582 0.706104 12.6507 0.674359L11.7536 0.409608C11.5297 0.343156 11.2939 0.471616 11.2279 0.695734L7.0632 14.7995C7.03145 14.9072 7.04373 15.023 7.09727 15.1214C7.15081 15.2202 7.2416 15.2932 7.34911 15.3252L8.24622 15.5899C8.28622 15.6018 8.32664 15.6075 8.36621 15.6075C8.54885 15.6075 8.71752 15.4881 8.77191 15.3038L12.9366 1.19984C12.9683 1.09212 12.9563 0.976357 12.9025 0.877949Z"
                                        fill="url(#paint2_linear_2838_1839)" />
                                    <defs>
                                        <linearGradient id="paint0_linear_2838_1839" x1="17.5312" y1="12.1666"
                                            x2="12.6806" y2="5.17616" gradientUnits="userSpaceOnUse">
                                            <stop stop-color="' . (activeMenu(route('user.codeTemplate'))['color1']) . '" />
                                            <stop offset="1" stop-color="' . activeMenu(route('user.codeTemplate'))['color2'] . '" />
                                        </linearGradient>
                                        <linearGradient id="paint1_linear_2838_1839" x1="4.58867" y1="12.166"
                                            x2="-0.264378" y2="5.1743" gradientUnits="userSpaceOnUse">
                                            <stop stop-color="' . activeMenu(route('user.codeTemplate'))['color1'] . '" />
                                            <stop offset="1" stop-color="' . activeMenu(route('user.codeTemplate'))['color2'] . '" />
                                        </linearGradient>
                                        <linearGradient id="paint2_linear_2838_1839" x1="10.8871" y1="13.7348"
                                            x2="3.81928" y2="7.54154" gradientUnits="userSpaceOnUse">
                                            <stop stop-color="' . activeMenu(route('user.codeTemplate'))['color1'] . '" />
                                            <stop offset="1" stop-color="' . activeMenu(route('user.codeTemplate'))['color2'] . '" />
                                        </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' => hasAccess('code') && customerPanelAccess('code'),
                'route' => route('user.codeTemplate'),
                'menu' => activeMenu(route('user.codeTemplate')),
                'settings' => [
                    'toggle_id' => 'code',
                    'label' => __('Code'),
                    'description' => __('Toggle to enable or disable code features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'speech_to_text',
                'category' => 'Voice & Audio',
                'name' => __('Speech to Text'),
                'description' => __('Convert spoken audio into accurate written text transcription.'),
                'icon' => '<span class="w-5 h-5 text-color-14 dark:text-white">
                                <svg width="20" height="17" viewBox="0 0 20 17" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M14.4718 0.562316C14.3031 0.598479 14.1526 0.693145 14.0471 0.829562C13.9415 0.965979 13.8875 1.13535 13.8948 1.3077V15.9231C13.8777 16.0277 13.8835 16.1347 13.9119 16.2368C13.9402 16.3389 13.9904 16.4336 14.059 16.5144C14.1276 16.5951 14.213 16.66 14.3092 16.7045C14.4053 16.749 14.51 16.772 14.616 16.772C14.7219 16.772 14.8266 16.749 14.9228 16.7045C15.019 16.66 15.1043 16.5951 15.1729 16.5144C15.2415 16.4336 15.2917 16.3389 15.3201 16.2368C15.3485 16.1347 15.3543 16.0277 15.3371 15.9231V1.3077C15.3396 1.20497 15.3203 1.10287 15.2806 1.00808C15.2409 0.913296 15.1817 0.827945 15.1068 0.757603C15.0319 0.687261 14.943 0.63351 14.8459 0.59986C14.7488 0.56621 14.6457 0.553417 14.5433 0.562316C14.5195 0.561148 14.4956 0.561148 14.4718 0.562316ZM5.24099 1.33155C5.07231 1.36771 4.92187 1.46238 4.81629 1.59879C4.7107 1.73521 4.65678 1.90458 4.66406 2.07693V15.1539C4.64693 15.2584 4.65274 15.3655 4.6811 15.4676C4.70945 15.5697 4.75967 15.6644 4.82827 15.7451C4.89688 15.8259 4.98222 15.8908 5.07839 15.9352C5.17456 15.9797 5.27925 16.0028 5.38522 16.0028C5.49118 16.0028 5.59587 15.9797 5.69204 15.9352C5.78821 15.8908 5.87356 15.8259 5.94216 15.7451C6.01076 15.6644 6.06098 15.5697 6.08933 15.4676C6.11769 15.3655 6.1235 15.2584 6.10637 15.1539V2.07693C6.10881 1.9742 6.08956 1.8721 6.04987 1.77731C6.01017 1.68253 5.95094 1.59718 5.87602 1.52683C5.80111 1.45649 5.7122 1.40274 5.6151 1.36909C5.518 1.33544 5.4149 1.32265 5.31252 1.33155C5.28869 1.33038 5.26482 1.33038 5.24099 1.33155ZM12.1641 3.63924C11.9954 3.6754 11.845 3.77007 11.7394 3.90648C11.6338 4.0429 11.5799 4.21227 11.5871 4.38462V12.8462C11.5696 12.9509 11.5752 13.0581 11.6033 13.1605C11.6315 13.2628 11.6816 13.3578 11.7502 13.4389C11.8188 13.5199 11.9042 13.585 12.0005 13.6296C12.0969 13.6743 12.2017 13.6974 12.3079 13.6974C12.4141 13.6974 12.519 13.6743 12.6153 13.6296C12.7116 13.585 12.797 13.5199 12.8656 13.4389C12.9342 13.3578 12.9843 13.2628 13.0125 13.1605C13.0407 13.0581 13.0462 12.9509 13.0287 12.8462V4.38462C13.0311 4.28203 13.0119 4.18008 12.9723 4.0854C12.9327 3.99073 12.8736 3.90546 12.7988 3.83514C12.7241 3.76483 12.6354 3.71104 12.5385 3.67729C12.4416 3.64353 12.3386 3.63057 12.2364 3.63924C12.212 3.63802 12.1884 3.63802 12.1641 3.63924ZM2.93329 4.40847C2.76462 4.44463 2.61418 4.5393 2.5086 4.67572C2.40301 4.81213 2.34909 4.9815 2.35637 5.15385V12.0769C2.33887 12.1816 2.34439 12.2889 2.37255 12.3913C2.40071 12.4936 2.45084 12.5886 2.51944 12.6696C2.58804 12.7506 2.67347 12.8157 2.76978 12.8604C2.86609 12.9051 2.97098 12.9282 3.07714 12.9282C3.1833 12.9282 3.28819 12.9051 3.3845 12.8604C3.48081 12.8157 3.56624 12.7506 3.63484 12.6696C3.70344 12.5886 3.75357 12.4936 3.78173 12.3913C3.80989 12.2889 3.81541 12.1816 3.79791 12.0769V5.15385C3.80033 5.05126 3.78111 4.94931 3.74151 4.85464C3.70191 4.75996 3.64282 4.67469 3.56807 4.60437C3.49333 4.53406 3.40461 4.48027 3.3077 4.44652C3.21079 4.41276 3.10786 4.3998 3.0056 4.40847C2.98126 4.40725 2.95764 4.40725 2.93329 4.40847ZM7.54868 5.1777C7.38001 5.21386 7.22957 5.30853 7.12398 5.44495C7.01839 5.58136 6.96447 5.75073 6.97175 5.92309V11.3077C6.95425 11.4124 6.95977 11.5197 6.98794 11.622C7.0161 11.7244 7.06622 11.8194 7.13482 11.9004C7.20342 11.9814 7.28885 12.0465 7.38516 12.0912C7.48148 12.1358 7.58636 12.159 7.69252 12.159C7.79869 12.159 7.90357 12.1358 7.99988 12.0912C8.0962 12.0465 8.18163 11.9814 8.25023 11.9004C8.31883 11.8194 8.36895 11.7244 8.39711 11.622C8.42527 11.5197 8.4308 11.4124 8.41329 11.3077V5.92309C8.41571 5.82049 8.3965 5.71854 8.3569 5.62387C8.3173 5.52919 8.2582 5.44392 8.18346 5.3736C8.10871 5.30329 8.02 5.2495 7.92309 5.21575C7.82617 5.18199 7.72324 5.16903 7.62099 5.1777C7.59664 5.17648 7.57302 5.17648 7.54868 5.1777ZM16.7794 5.1777C16.6108 5.21386 16.4603 5.30853 16.3547 5.44495C16.2492 5.58136 16.1952 5.75073 16.2025 5.92309V11.3077C16.1854 11.4123 16.1912 11.5193 16.2196 11.6214C16.2479 11.7235 16.2981 11.8182 16.3667 11.899C16.4353 11.9797 16.5207 12.0446 16.6169 12.0891C16.713 12.1336 16.8177 12.1566 16.9237 12.1566C17.0296 12.1566 17.1343 12.1336 17.2305 12.0891C17.3267 12.0446 17.412 11.9797 17.4806 11.899C17.5492 11.8182 17.5994 11.7235 17.6278 11.6214C17.6562 11.5193 17.662 11.4123 17.6448 11.3077V5.92309C17.6473 5.82035 17.628 5.71826 17.5883 5.62347C17.5486 5.52868 17.4894 5.44333 17.4145 5.37299C17.3396 5.30265 17.2507 5.24889 17.1536 5.21524C17.0565 5.18159 16.9534 5.1688 16.851 5.1777C16.8272 5.17653 16.8033 5.17653 16.7794 5.1777ZM0.625601 6.71616C0.456929 6.75233 0.306489 6.84699 0.200903 6.98341C0.0953174 7.11983 0.0413947 7.2892 0.0486779 7.46155V9.76924C0.0315452 9.87381 0.037358 9.98085 0.0657132 10.0829C0.0940684 10.185 0.144288 10.2798 0.21289 10.3605C0.281492 10.4413 0.366836 10.5061 0.463006 10.5506C0.559175 10.5951 0.66387 10.6182 0.769832 10.6182C0.875794 10.6182 0.980488 10.5951 1.07666 10.5506C1.17283 10.5061 1.25817 10.4413 1.32677 10.3605C1.39538 10.2798 1.4456 10.185 1.47395 10.0829C1.50231 9.98085 1.50812 9.87381 1.49099 9.76924V7.46155C1.49343 7.35881 1.47417 7.25672 1.43448 7.16193C1.39479 7.06714 1.33555 6.98179 1.26064 6.91145C1.18572 6.84111 1.09681 6.78736 0.999717 6.75371C0.902619 6.72006 0.799516 6.70726 0.697139 6.71616C0.673052 6.71497 0.649689 6.71497 0.625601 6.71616ZM9.85637 6.71616C9.6877 6.75233 9.53726 6.84699 9.43167 6.98341C9.32609 7.11983 9.27216 7.2892 9.27945 7.46155V9.76924C9.26195 9.87395 9.26747 9.98121 9.29563 10.0836C9.32379 10.1859 9.37391 10.2809 9.44251 10.3619C9.51111 10.443 9.59654 10.5081 9.69286 10.5527C9.78917 10.5974 9.89406 10.6205 10.0002 10.6205C10.1064 10.6205 10.2113 10.5974 10.3076 10.5527C10.4039 10.5081 10.4893 10.443 10.5579 10.3619C10.6265 10.2809 10.6766 10.1859 10.7048 10.0836C10.733 9.98121 10.7385 9.87395 10.721 9.76924V7.46155C10.7234 7.35895 10.7042 7.257 10.6646 7.16233C10.625 7.06766 10.5659 6.98238 10.4912 6.91207C10.4164 6.84175 10.3277 6.78796 10.2308 6.75421C10.1339 6.72046 10.0309 6.70749 9.92868 6.71616C9.90433 6.71494 9.88071 6.71494 9.85637 6.71616ZM19.0871 6.71616C18.9185 6.75233 18.768 6.84699 18.6624 6.98341C18.5569 7.11983 18.5029 7.2892 18.5102 7.46155V9.76924C18.4931 9.87381 18.4989 9.98085 18.5273 10.0829C18.5556 10.185 18.6058 10.2798 18.6744 10.3605C18.743 10.4413 18.8284 10.5061 18.9245 10.5506C19.0207 10.5951 19.1254 10.6182 19.2314 10.6182C19.3373 10.6182 19.442 10.5951 19.5382 10.5506C19.6344 10.5061 19.7197 10.4413 19.7883 10.3605C19.8569 10.2798 19.9071 10.185 19.9355 10.0829C19.9638 9.98085 19.9697 9.87381 19.9525 9.76924V7.46155C19.955 7.35881 19.9357 7.25672 19.896 7.16193C19.8563 7.06714 19.7971 6.98179 19.7222 6.91145C19.6473 6.84111 19.5584 6.78736 19.4613 6.75371C19.3642 6.72006 19.2611 6.70726 19.1587 6.71616C19.1348 6.71499 19.111 6.71499 19.0871 6.71616Z" fill="url(#paint0_linear_9514_4841)"/>
                                    <defs>
                                    <linearGradient id="paint0_linear_9514_4841" x1="12.9925" y1="14.7766" x2="8.1441" y2="1.33125" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . activeMenu(route('user.speechTemplate'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.speechTemplate'))['color2'] . '"/>
                                    </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' => hasAccess('speech_to_text') && customerPanelAccess('speech_to_text'),
                'route' => route('user.speechTemplate'),
                'menu' => activeMenu(route('user.speechTemplate')),
                'settings' => [
                    'toggle_id' => 'speech_to_text',
                    'label' => __('Speech to Text'),
                    'description' => __('Toggle to enable or disable speech-to-text features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'voiceover',
                'category' => 'Voice & Audio',
                'name' => __('Voiceover'),
                'description' => __('Transform written text into natural-sounding voice narration.'),
                'icon' => '<span class="w-5 h-5">
                                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="18" viewBox="0 0 14 18" fill="none">
                                    <path d="M7 11.1875C8.83398 11.1875 10.3203 9.71875 10.3203 7.90625V3.53125C10.3203 1.71875 8.83398 0.25 7 0.25C5.16602 0.25 3.67969 1.71875 3.67969 3.53125V7.90625C3.67969 9.71875 5.16602 11.1875 7 11.1875ZM13.4453 7.86719C13.4453 7.78125 13.375 7.71094 13.2891 7.71094H12.1172C12.0312 7.71094 11.9609 7.78125 11.9609 7.86719C11.9609 10.6074 9.74023 12.8281 7 12.8281C4.25977 12.8281 2.03906 10.6074 2.03906 7.86719C2.03906 7.78125 1.96875 7.71094 1.88281 7.71094H0.710938C0.625 7.71094 0.554688 7.78125 0.554688 7.86719C0.554688 11.1621 3.02734 13.8809 6.21875 14.2656V16.2656H3.38086C3.11328 16.2656 2.89844 16.5449 2.89844 16.8906V17.5938C2.89844 17.6797 2.95313 17.75 3.01953 17.75H10.9805C11.0469 17.75 11.1016 17.6797 11.1016 17.5938V16.8906C11.1016 16.5449 10.8867 16.2656 10.6191 16.2656H7.70312V14.2754C10.9316 13.9238 13.4453 11.1895 13.4453 7.86719Z" fill="url(#paint0_linear_9790_5737)"/>
                                    <defs>
                                      <linearGradient id="paint0_linear_9790_5737" x1="8.9358" y1="15.5961" x2="1.69141" y2="3.55391" gradientUnits="userSpaceOnUse">
                                        <stop stop-color="' . activeMenu(route('user.voicoverTemplate'))['color1'] . '"/>
                                        <stop offset="1" stop-color="' . activeMenu(route('user.voicoverTemplate'))['color2'] . '"/>
                                      </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' =>  hasAccess('voiceover') && customerPanelAccess('voiceover'),
                'route' => route('user.voicoverTemplate'),
                'menu' => activeMenu(route('user.voicoverTemplate')),
                'settings' => [
                    'toggle_id' => 'voiceover',
                    'label' => __('Voiceover'),
                    'description' => __('Toggle to enable or disable voiceover features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'chat',
                'category' => 'AI Conversational Tools',
                'name' => __('Chat'),
                'description' => __('Interactive conversational AI for questions, brainstorming, and assistance.'),
                'icon' => '<svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M1.66797 18.3332V3.33317C1.66797 2.87484 1.8313 2.48262 2.15797 2.1565C2.48464 1.83039 2.87686 1.66706 3.33464 1.6665H16.668C17.1263 1.6665 17.5188 1.82984 17.8455 2.1565C18.1721 2.48317 18.3352 2.87539 18.3346 3.33317V13.3332C18.3346 13.7915 18.1716 14.184 17.8455 14.5107C17.5194 14.8373 17.1269 15.0004 16.668 14.9998H5.0013L1.66797 18.3332ZM5.0013 11.6665H11.668V9.99984H5.0013V11.6665ZM5.0013 9.1665H15.0013V7.49984H5.0013V9.1665ZM5.0013 6.6665H15.0013V4.99984H5.0013V6.6665Z" fill="url(#paint0_linear_14102_7464)"/>
                                <defs>
                                <linearGradient id="paint0_linear_14102_7464" x1="12.5042" y1="16.2818" x2="6.71879" y2="3.22618" gradientUnits="userSpaceOnUse">
                                <stop stop-color="' . activeMenu(route('chat.index'))['color1'] . '"/>
                                <stop offset="1" stop-color="' . activeMenu(route('chat.index'))['color2'] . '"/>
                                </linearGradient>
                                </defs>
                                </svg>',
                'access' => hasAccess('chat') && (customerPanelAccess('chat') && (\Modules\Addons\Entities\Addon::find('Chat')->isEnabled())),
                'route' => route('chat.index'),
                'menu' => activeMenu(route('chat.index')),
                'settings' => [
                    'toggle_id' => 'chat',
                    'label' => __('Chat'),
                    'description' => __('Toggle to enable or disable chat features.'),
                ],
                'type' => 'feature',
                'isReactRouter' => true
            ],
            [
                'id' => 'chatbot',
                'category' => 'AI Conversational Tools',
                'name' => __('AI Chatbot'),
                'description' => __('Deploy intelligent chatbots for customer service and engagement.'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" version="1.1" xmlns:xlink="http://www.w3.org/1999/xlink" width="20" height="20" x="0" y="0" viewBox="0 0 32 32" style="enable-background:new 0 0 512 512" xml:space="preserve" fill-rule="evenodd" class=""><g><linearGradient id="a" x1="0" x2="1" y1="0" y2="0" gradientTransform="matrix(15 23.4 -15.6 22.5 334.2 116)" gradientUnits="userSpaceOnUse"><stop stop-opacity="1" stop-color="#141414" offset="0"></stop><stop stop-opacity="1" stop-color="#141414" offset="1"></stop></linearGradient><path fill="url(#a)" d="M333.6 137.446c-.699-.37-1.2-1.37-1.2-2.546v-3.6c0-1.176.501-2.176 1.2-2.546zm3-13.346v-1.914c-1.035-.4-1.8-1.81-1.8-3.486 0-1.987 1.075-3.6 2.4-3.6s2.4 1.613 2.4 3.6c0 1.676-.765 3.086-1.8 3.486v1.914zm6 0c-.994 0-1.8-1.209-1.8-2.7v-3.6c0-1.491.806-2.7 1.8-2.7h4.8c.994 0 1.8 1.209 1.8 2.7v3.6c0 1.491-.806 2.7-1.8 2.7h-.879l-.203.457c-.407-.298-.857-.457-1.318-.457zm5.4 4.654c.699.37 1.2 1.37 1.2 2.546v3.6c0 1.176-.501 2.176-1.2 2.546zm-3-2.854c.477 0 .935.284 1.273.791.337.506.527 1.193.527 1.909v9c0 .716-.19 1.403-.527 1.909-.338.507-.796.791-1.273.791h-8.4c-.477 0-.935-.284-1.273-.791-.337-.506-.527-1.193-.527-1.909v-9c0-.716.19-1.403.527-1.909.338-.507.796-.791 1.273-.791zm-2.4 6.3v1.8c0 .497.269.9.6.9s.6-.403.6-.9v-1.8c0-.497-.269-.9-.6-.9s-.6.403-.6.9zm-4.8 0v1.8c0 .497.269.9.6.9s.6-.403.6-.9v-1.8c0-.497-.269-.9-.6-.9s-.6.403-.6.9z" transform="matrix(1.667 0 0 1.111 -552 -126.889)" opacity="1" data-original="url(#a)" class=""></path></g></svg>',
                'access' => hasAccess('aichatbot') && (customerPanelAccess('chatbot') && (\Modules\Addons\Entities\Addon::find('Chatbot')->isEnabled())),
                'route' => route('aichatbot.index'),
                'menu' => activeMenu(route('aichatbot.index')),
                'settings' => [
                    'toggle_id' => 'aichatbot',
                    'label' => __('AI Chatbot'),
                    'description' => __('Toggle to enable or disable the AI chatbot feature.'),
                ],
                'type' => 'feature',
                'isReactRouter' => true
            ],
            [
                'id' => 'plagiarism',
                'category' => 'Content Analysis',
                'name' => __('Plagiarism'),
                'description' => __('Detect and analyze content originality and potential plagiarism.'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" id="Layer_1" enable-background="new 0 0 510 510" height="20px" viewBox="0 0 510 510" width="20px">
                                <defs>
                                    <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%" gradientUnits="userSpaceOnUse">
                                    <stop offset="0%" stop-color="' . activeMenu(route('user.plagiarismTemplate'))['color1'] . '" />
                                    <stop offset="100%" stop-color="' . activeMenu(route('user.plagiarismTemplate'))['color2'] . '" />
                                    </linearGradient>
                                </defs>
                                <g fill="url(#gradient)">
                                    <path d="m366.194 143.975h68.862l-68.847-69.11z"/>
                                    <path d="m411.213.189h-201.213v29.623h153.673l116.327 116.77v243.607h30v-291.214z"/>
                                    <path d="m65.745 401.64-56.959 56.959c-11.715 11.716-11.715 30.71 0 42.426 11.716 11.716 30.711 11.716 42.427 0l56.959-56.958c-17.108-10.794-31.633-25.32-42.427-42.427z"/>
                                    <path d="m336.213 59.812h-171.213v135.838c134.539-13.885 203.422 154.842 99.748 239.162h185.252v-260.837h-113.812zm23.787 180.377h60v30h-60zm0 60h60v30h-60zm0 60h60v30h-60z"/>
                                    <path d="m180 299.812c-16.542 0-30 13.458-30 30 1.648 39.799 58.358 39.787 60 0 0-16.542-13.458-30-30-30z"/>
                                    <path d="m180 224.812c-57.99 0-105 47.01-105 105 5.53 139.28 204.491 139.241 210-.001 0-57.99-47.01-104.999-105-104.999zm0 165c-33.084 0-60-26.916-60-60 3.296-79.598 116.716-79.575 120 0 0 33.084-26.916 60-60 60z"/>
                                </g>
                            </svg>',
                'access' => hasAccess('plagiarism') && customerPanelAccess('plagiarism'),
                'route' => route('user.plagiarismTemplate'),
                'menu' => activeMenu(route('user.plagiarismTemplate')),
                'settings' => [
                    'toggle_id' => 'plagiarism',
                    'label' => __('Plagiarism'),
                    'description' => __('Toggle to enable or disable plagiarism checking features.'),
                ],
                'type' => 'feature',
            ],
            [
                'id' => 'ai_detector',
                'category' => 'Content Analysis',
                'name' => __('Ai Detector'),
                'description' => __('Identify and analyze AI-generated content with advanced detection.'),
                'icon' => '<svg id="glipy_copy" viewBox="0 0 64 64" width="22" height="22" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" data-name="glipy copy">
                                <linearGradient id="linear-gradient" gradientUnits="userSpaceOnUse" x1="4.061" x2="59.939" y1="32" y2="32">
                                    <stop offset="0" stop-color="' . activeMenu(route('user.aiDetectorTemplate'))['color1'] . '"/>
                                    <stop offset="1" stop-color="' . activeMenu(route('user.aiDetectorTemplate'))['color2'] . '"/>
                                </linearGradient>
                                <path d="m26.50909 44.25943-14.34948 14.33941a4.68041 4.68041 0 0 1 -3.34972 1.39991 4.77865 4.77865 0 0 1 -3.36-8.11962l14.3294-14.33947c.33.5.67.98 1.02 1.42994a22.71088 22.71088 0 0 0 5.7098 5.28983zm12.53949 1.73993a20.99939 20.99939 0 0 1 -17.819-9.90953c-8.74344-13.64565 1.50742-32.24648 17.819-32.08883 27.85505 1.1501 27.85328 40.84868 0 41.99836zm16.49942-20.99906c-.9003-21.88288-32.09519-21.8893-32.99878-.00012a16.49939 16.49939 0 0 0 32.99878.00012zm-23.63916-7.12985h14.26946v14.26946h-14.26946zm9.93962 11.00955a1 1 0 0 0 1.99992 0v-7.7497a1.00007 1.00007 0 0 0 -1.99992 0zm-8.16967 0a1.00008 1.00008 0 0 0 1.99992.00006v-2.87992h2.83986v2.87986a1 1 0 1 0 1.99992 0v-5.81975a2.934 2.934 0 0 0 -2.92988-2.92988h-.96993a2.94257 2.94257 0 0 0 -2.93989 2.92988zm12.46949-16.50934v3.49987h1.03a.99025.99025 0 0 1 1 1v1.02993h1.14a1.00012 1.00012 0 0 1 -.00006 1.99992h-1.13999v2.06992h3.17987a1.00008 1.00008 0 0 1 -.00007 1.99993h-3.1798v2.06992h2.0499a1.00012 1.00012 0 0 1 -.00006 1.99985h-2.04984v2.06h4.41977a14.55356 14.55356 0 0 0 -6.44972-17.72934zm-16.23936 4.49983a.99651.99651 0 0 1 1-1h1.04v-1.14a1 1 0 0 1 1.99992 0v1.14h2.05992v-3.17983a1.00007 1.00007 0 0 1 1.99992 0v3.17987h2.06993v-2.04991a1 1 0 0 1 1.99992 0v2.04991h2.06986v-4.41981a14.55323 14.55323 0 0 0 -17.7292 6.44988l3.48973-.00018zm0 16.26938v-1.04h-1.14a1.00015 1.00015 0 0 1 .00008-1.99987h1.13991v-2.06h-3.16985a1.00007 1.00007 0 0 1 .00006-1.99992h3.1698v-2.06989h-2.04992a1 1 0 0 1 0-1.99993h2.04991v-2.06992h-4.4098a14.00423 14.00423 0 0 0 -.95 5.09984 14.49959 14.49959 0 0 0 7.39974 12.62951v-3.48986h-1.04a1.00292 1.00292 0 0 1 -.99993-.99996zm18.26931 0a.99651.99651 0 0 1 -1 1h-1.03v1.13991a1 1 0 0 1 -1.99993 0v-1.13995h-2.06981v3.16986a1.00007 1.00007 0 0 1 -1.99992-.00006v-3.1698h-2.06993v2.04991a1 1 0 0 1 -1.99992 0v-2.04991h-2.05992v4.41981a14.57451 14.57451 0 0 0 17.7292-6.45995l-3.49974.00018zm-9.65966-10.07962a.93463.93463 0 0 0 -.93-.93h-.96993a.93677.93677 0 0 0 -.94.93v.94h2.83986z" fill="url(#linear-gradient)"/></svg>',
                'access' => hasAccess('ai_detector') && customerPanelAccess('ai_detector'),
                'route' => route('user.aiDetectorTemplate'),
                'menu' => activeMenu(route('user.aiDetectorTemplate')),
                'settings' => [
                    'toggle_id' => 'ai_detector',
                    'label' => __('AI Detector'),
                    'description' => __('Toggle to enable or disable the AI Detector feature.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'voice_clone',
                'category' => 'Voice & Audio',
                'name' => __('Voice Clone'),
                'description' => __('Create realistic voice clones and personalized speech synthesis.'),
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64" width="22" height="22">
                        <defs>
                            <linearGradient id="voiceCloneGradient" x1="0%" y1="50%" x2="100%" y2="50%" gradientUnits="userSpaceOnUse">
                                <stop offset="0" stop-color="' . (activeMenu(route('user.voiceClone.template'), route('user.voiceClone.edit', ['id' => $id]))['color1'] ?? $color1) . '" />
                                <stop offset="1" stop-color="' . (activeMenu(route('user.voiceClone.template'), route('user.voiceClone.edit', ['id' => $id]))['color2'] ?? $color2) . '" />
                            </linearGradient>
                        </defs>
                        <circle cx="32" cy="32" r="30" fill="url(#voiceCloneGradient)" />
                        <g class="dark:fill-[#2C2C2C] fill-white" transform="translate(32,32)">
                            <rect x="-4" y="-16" width="8" height="20" rx="4" />
                            <rect x="-2" y="4" width="4" height="6" rx="2" />
                            <rect x="-6" y="10" width="12" height="3" rx="1.5" />
                        </g>
                        <g class="dark:stroke-white stroke-[#2C2C2C]" stroke-width="2.5" stroke-linecap="round" fill="none">
                            <line x1="18" y1="26" x2="18" y2="38" />
                            <line x1="24" y1="22" x2="24" y2="42" />
                            <line x1="30" y1="18" x2="30" y2="46" />
                            <line x1="36" y1="22" x2="36" y2="42" />
                            <line x1="42" y1="26" x2="42" y2="38" />
                        </g>
                    </svg>',
                'access' => hasAccess('voice_clone') && customerPanelAccess('voice_clone'),
                'route' => route('user.voiceClone.template'),
                'menu' =>  activeMenu(route('user.voiceClone.template'), route('user.voiceClone.edit', ['id' => $id])),
                'settings' => [
                    'toggle_id' => 'voice_clone',
                    'label' => __('Voice Clone'),
                    'description' => __('Toggle to enable or disable the Voice Clone feature.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'ai_persona',
                'category' => 'Video Studio',
                'name' => __('AI Persona'),
                'description' => __('Create and customize AI personalities for specialized interactions.'),
                'icon' => '<svg stroke-width="1.5" class="size-navbar-icon" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke-linecap="round" stroke-linejoin="round">
                                    <!-- Gradient Definition -->
                                    <defs>
                                        <linearGradient id="aiPersonaGradient" x1="0" y1="12" x2="24" y2="12" gradientUnits="userSpaceOnUse">
                                        <stop offset="0" stop-color="' .  activeMenu(route('user.ai-persona.template'))['color1'] . '" />
                                        <stop offset="1" stop-color="' .  activeMenu(route('user.ai-persona.template'))['color2'] . '" />
                                        </linearGradient>
                                    </defs>
                                    <!-- Icon Paths with Gradient Stroke -->
                                    <path d="M10.5 20h-5.5a2 2 0 0 1 -2 -2v-9a2 2 0 0 1 2 -2h1a2 2 0 0 0 2 -2a1 1 0 0 1 1 -1h6a1 1 0 0 1 1 1a2 2 0 0 0 2 2h1a2 2 0 0 1 2 2v2.5" stroke="url(#aiPersonaGradient)" />
                                    <path d="M14.569 11.45a3 3 0 1 0 -4.518 3.83" stroke="url(#aiPersonaGradient)" />
                                    <path d="M17.8 20.817l-2.172 1.138a.392 .392 0 0 1 -.568 -.41l.415 -2.411l-1.757 -1.707a.389 .389 0 0 1 .217 -.665l2.428 -.352l1.086 -2.193a.392 .392 0 0 1 .702 0l1.086 2.193l2.428 .352a.39 .39 0 0 1 .217 .665l-1.757 1.707l.414 2.41a.39 .39 0 0 1 -.567 .411l-2.172 -1.138z" stroke="url(#aiPersonaGradient)" />
                                </svg>',
                'access' => hasAccess('ai_persona') && customerPanelAccess('video'),
                'route' => route('user.ai-persona.template'),
                'menu' => activeMenu(route('user.ai-persona.template')),
                'settings' => [
                    'toggle_id' => 'ai_persona',
                    'label' => __('AI Persona'),
                    'description' => __('Toggle to enable or disable AI Persona features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'ai_avatar',
                'category' => 'Video Studio',
                'name' => __('AI Avatar'),
                'description' => __('Generate personalized avatars and digital representations.'),
                'icon' => '<svg stroke-width="1.5" class="size-navbar-icon" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 64 64" fill="nonestroke-linecap="round" stroke-linejoin="round">
                                    <!-- Gradient Definition -->
                                    <defs>
                                        <linearGradient id="aiAvatarGradient" x1="0" y1="12" x2="24" y2="12" gradientUnits="userSpaceOnUse">
                                            <stop offset="0" stop-color="' . activeMenu(route('user.ai-avatar.template'))['color1'] . '" />
                                            <stop offset="1" stop-color="' . activeMenu(route('user.ai-avatar.template'))['color2'] . '" />
                                        </linearGradient>
                                    </defs>
                                    <!-- Icon Paths with Gradient Stroke -->
                                    <g>
                                        <path d="M48.11,47.066L38,45.17v-1.649c2.97-1.553,5.266-4.218,6.335-7.442L45.414,35h-0.777c0.085-0.356,0.152-0.718,0.206-1.085 C47.198,33.511,49,31.468,49,29c0-1.978-1.164-3.676-2.835-4.486C49.005,23.742,51,22.613,51,21c0-1.631-1.963-2.863-5.937-3.762 c0.14-4.249,0.608-8.496,1.441-12.66L47.22,1H16.78l0.716,3.579c0.833,4.163,1.302,8.411,1.441,12.66 C14.963,18.137,13,19.369,13,21c0,1.613,1.995,2.742,4.835,3.514C16.164,25.324,15,27.022,15,29c0,2.468,1.802,4.511,4.157,4.915 c0.055,0.367,0.122,0.729,0.206,1.085h-0.777l1.079,1.079c1.069,3.224,3.364,5.888,6.335,7.442v1.649l-10.11,1.896 c-3.495,0.654-6.471,2.973-7.961,6.201L3.437,63h57.126l-4.492-9.732C54.581,50.039,51.605,47.721,48.11,47.066z M47,29 c0,1.302-0.839,2.402-2,2.816v-5.631C46.161,26.598,47,27.698,47,29z M44.78,3l-0.237,1.186c-0.656,3.281-1.085,6.613-1.323,9.956 C41.903,14.397,38.145,15,32,15c-6.145,0-9.903-0.603-11.22-0.858c-0.238-3.344-0.667-6.676-1.323-9.956L19.22,3H44.78z M43.095,16.201c-0.045,0.994-0.086,1.989-0.093,2.983C41.511,19.458,37.822,20,32,20s-9.511-0.542-11.002-0.816 c-0.007-0.994-0.048-1.989-0.093-2.983C22.631,16.505,26.309,17,32,17C37.691,17,41.369,16.505,43.095,16.201z M19,31.816 c-1.161-0.414-2-1.514-2-2.816s0.839-2.402,2-2.816V31.816z M14.998,21.019c0.07-0.24,0.972-1.017,3.99-1.747 C18.989,19.437,19,19.603,19,19.769v1.012l0.757,0.189C19.926,21.012,23.969,22,32,22s12.074-0.988,12.243-1.03L45,20.781v-1.012 c0-0.166,0.011-0.331,0.012-0.497c3.017,0.729,3.92,1.503,3.99,1.711C48.706,22.036,42.605,24,32,24S15.294,22.036,14.998,21.019z M21.422,34.984C21.154,34.033,21,33.035,21,32v-6.816C24.63,25.784,28.812,26,32,26s7.37-0.216,11-0.816V32 c0,1.035-0.154,2.033-0.422,2.984c-1.439-0.042-2.788-0.608-3.81-1.63C37.894,32.48,36.734,32,35.5,32s-2.394,0.48-3.268,1.354 L32,33.586l-0.232-0.232C30.894,32.48,29.734,32,28.5,32s-2.394,0.48-3.268,1.354C24.211,34.375,22.861,34.942,21.422,34.984z M23.173,36.759c1.303-0.336,2.498-1.015,3.474-1.991c0.991-0.99,2.716-0.99,3.707,0L32,36.414l1.646-1.646 c0.991-0.99,2.716-0.99,3.707,0c0.976,0.977,2.17,1.655,3.474,1.991l-0.767,0.767C39.109,38.477,37.845,39,36.5,39 s-2.609-0.523-3.561-1.475L32,36.586l-0.939,0.939C30.109,38.477,28.845,39,27.5,39s-2.609-0.523-3.561-1.475L23.173,36.759z M25.148,40.59C25.894,40.855,26.685,41,27.5,41c1.664,0,3.239-0.573,4.5-1.626C33.261,40.427,34.836,41,36.5,41 c0.815,0,1.606-0.145,2.352-0.41C36.971,42.094,34.591,43,32,43S27.029,42.094,25.148,40.59z M6.563,61l3.182-6.895 c1.219-2.641,3.654-4.538,6.513-5.074l6.111-1.146l-2.939,1.959L22.586,53l-2.495,2.494l3.027,0.866 c2.771,0.792,5.179,2.419,6.954,4.64H6.563z M31,58.995c-1.903-2.098-4.338-3.654-7.093-4.488L25.414,53l-2.844-2.845l4.269-2.846 C29.518,50.268,31,54.051,31,58.071V58.995z M28,45.616v-1.252C29.261,44.773,30.604,45,32,45s2.739-0.227,4-0.636v1.252 c-1.816,1.904-3.165,4.135-4,6.561C31.165,49.751,29.816,47.52,28,45.616z M33,58.995v-0.923c0-4.02,1.482-7.804,4.161-10.762 l4.269,2.846L38.586,53l1.507,1.507C37.338,55.34,34.903,56.897,33,58.995z M33.928,61c1.775-2.22,4.183-3.848,6.954-4.64 l3.027-0.866L41.414,53l3.156-3.155l-2.939-1.959l6.111,1.146c2.859,0.536,5.294,2.434,6.513,5.074L57.437,61H33.928z" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" width="2" x="56" y="5" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" transform="matrix(0.7071 -0.7071 0.7071 0.7071 10.2347 40.345)" width="2" x="52.818" y="6.318" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="3" x="51" y="10" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" transform="matrix(0.7071 -0.7071 0.7071 0.7071 5.7349 42.2089)" width="3" x="52.318" y="13.182" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" width="2" x="56" y="14" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" transform="matrix(0.7071 -0.7071 0.7071 0.7071 7.5988 46.7088)" width="2" x="59.182" y="12.682" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="3" x="60" y="10" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" transform="matrix(0.7071 -0.7071 0.7071 0.7071 12.0986 44.8449)" width="3" x="58.682" y="6.818" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" width="2" x="6" y="34" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" transform="matrix(0.707 -0.7072 0.7072 0.707 -24.9189 13.487)" width="2" x="2.818" y="35.318" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="3" x="1" y="39" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" transform="matrix(0.7071 -0.7071 0.7071 0.7071 -29.4157 15.3474)" width="3" x="2.318" y="42.182" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" width="2" x="6" y="43" stroke="url(#aiAvatarGradient)" />
                                        <rect height="3" transform="matrix(0.707 -0.7072 0.7072 0.707 -27.5545 19.8516)" width="2" x="9.182" y="41.682" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="3" x="10" y="39" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" transform="matrix(0.7071 -0.7071 0.7071 0.7071 -23.0523 17.9835)" width="3" x="8.682" y="35.818" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="53" y="31" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="53" y="35" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="55" y="33" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="51" y="33" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="11" y="9" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="11" y="13" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="13" y="11" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="9" y="11" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="3" y="20" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="3" y="24" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="5" y="22" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="1" y="22" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="59" y="44" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="59" y="48" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="61" y="46" stroke="url(#aiAvatarGradient)" />
                                        <rect height="2" width="2" x="57" y="46" stroke="url(#aiAvatarGradient)" />
                                    </g>
                                </svg>',
                'access' => hasAccess('ai_avatar') && customerPanelAccess('video'),
                'route' => route('user.ai-avatar.template'),
                'menu' => activeMenu(route('user.ai-avatar.template')),
                'settings' => [
                    'toggle_id' => 'ai_avatar',
                    'label' => __('AI Avatar'),
                    'description' => __('Toggle to enable or disable AI Avatar features.'),
                ],
                'type' => 'feature'
            ],
            [
                'id' => 'ai_product_photography',
                'category' => 'Visual Studio',
                'name' => __('Ai Product Photography'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<svg stroke-width="1.5" class="size-navbar-icon" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke-linecap="round" stroke-linejoin="round">
                        <!-- Gradient Definition -->
                        <defs>
                            <linearGradient id="customGradient" x1="0" y1="12" x2="24" y2="12" gradientUnits="userSpaceOnUse">
                            <stop offset="0" stop-color="' . (activeMenu(route('user.ai-product-photography.template'))['color1'] ?? $color1) . '" />
                            <stop offset="1" stop-color="' . (activeMenu(route('user.ai-product-photography.template'))['color2'] ?? $color2) . '" />
                            </linearGradient>
                        </defs>

                        <!-- Icon Paths with Gradient Stroke -->
                        <path d="M15 8h.01" stroke="url(#customGradient)"></path>
                        <path d="M11 21h-5a3 3 0 0 1 -3 -3v-12a3 3 0 0 1 3 -3h12a3 3 0 0 1 3 3v5.5" stroke="url(#customGradient)"></path>
                        <path d="M3 16l5 -5c.928 -.893 2.072 -.893 3 0l2 2" stroke="url(#customGradient)"></path>
                        <path d="M17.8 20.817l-2.172 1.138a.392 .392 0 0 1 -.568 -.41l.415 -2.411l-1.757 -1.707a.389 .389 0 0 1 .217 -.665l2.428 -.352l1.086 -2.193a.392 .392 0 0 1 .702 0l1.086 2.193l2.428 .352a.39 .39 0 0 1 .217 .665l-1.757 1.707l.414 2.41a.39 .39 0 0 1 -.567 .411l-2.172 -1.138z" stroke="url(#customGradient)"></path>
                    </svg>',
                'access' => hasAccess('ai_product_photography') && customerPanelAccess('image'),
                'route' => route('user.ai-product-photography.template'),
                'menu' => activeMenu(route('user.ai-product-photography.template')),
                'settings' => [
                    'toggle_id' => 'ai_product_photography',
                    'label' => __('Ai Product Photography'),
                    'description' => __('Toggle to enable or disable AI Avatar features.'),
                ],
                'type' => 'feature'
            ],
        ];

        $features = apply_filters('modify_feature_data', $features);

        $commonFeatures = [
            [
                'id' => 'ticket',
                'category' => 'System Management',
                'name' => __('Support Ticket'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 20 20" fill="none">
                                    <g clip-path="url(#clip0_8222_4651)">
                                      <path d="M19.375 8.4375V11.5625C19.375 12.4243 18.6743 13.125 17.8125 13.125H17.3871C16.7896 16.6663 13.7091 19.375 10 19.375C9.65454 19.375 9.375 19.0955 9.375 18.75C9.375 18.4045 9.65454 18.125 10 18.125C13.446 18.125 16.25 15.321 16.25 11.875V8.125C16.25 4.67896 13.446 1.875 10 1.875C6.55396 1.875 3.75 4.67896 3.75 8.125V12.5C3.75 12.8455 3.47046 13.125 3.125 13.125H2.1875C1.32568 13.125 0.625 12.4243 0.625 11.5625V8.4375C0.625 7.57568 1.32568 6.875 2.1875 6.875H2.61292C3.21045 3.33374 6.29089 0.625 10 0.625C13.7091 0.625 16.7896 3.33374 17.3871 6.875H17.8125C18.6743 6.875 19.375 7.57568 19.375 8.4375ZM10 3.125C7.24304 3.125 5 5.36804 5 8.125V11.25C5 14.007 7.24304 16.25 10 16.25C12.757 16.25 15 14.007 15 11.25V8.125C15 5.36804 12.757 3.125 10 3.125Z" fill="url(#paint0_linear_8222_4651)"/>
                                    </g>
                                    <defs>
                                      <linearGradient id="paint0_linear_8222_4651" x1="12.8157" y1="17.0672" x2="6.30717" y2="2.37963" gradientUnits="userSpaceOnUse">
                                        <stop stop-color="' . (activeMenu(route('user.ticketList'), route('user.searchList'), route('user.ticketAdd'), route('user.threadReply', ['id' => $id] ))['color1'] ?? $color1) . '"/>
                                        <stop offset="1" stop-color="' . (activeMenu(route('user.ticketList'), route('user.searchList'), route('user.ticketAdd'), route('user.threadReply', ['id' => $id] ))['color2'] ?? $color2) . '"/>
                                      </linearGradient>
                                      <clipPath id="clip0_8222_4651">
                                        <rect width="20" height="20" fill="white"/>
                                      </clipPath>
                                    </defs>
                                  </svg>
                            </span>',
                'access' => true,
                'route' => route('user.ticketList'),
                'menu' => activeMenu(route('user.ticketList'), route('user.searchList'), route('user.ticketAdd'), route('user.threadReply', ['id' => $id] )),
                'type' => 'common'
            ],
            [
                'id' => 'drive',
                'category' => 'System Management',
                'name' => __('Drive'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M11.057 3.375H15.5362C16.8593 3.375 17.8967 4.51107 17.7769 5.82871L17.061 13.7037C16.9557 14.8626 15.984 15.75 14.8203 15.75H3.17975C2.01604 15.75 1.04434 14.8626 0.938986 13.7037L0.223077 5.82871C0.173514 5.28352 0.322075 4.76941 0.607013 4.354L0.562513 3.375C0.562513 2.13236 1.56987 1.125 2.81251 1.125H6.94303C7.53977 1.125 8.11207 1.36205 8.53402 1.78401L9.466 2.71599C9.88796 3.13795 10.4603 3.375 11.057 3.375ZM1.69479 3.5096C1.93419 3.42259 2.19303 3.375 2.46384 3.375H8.53402L7.73853 2.57951C7.52755 2.36853 7.2414 2.25 6.94303 2.25H2.81251C2.19829 2.25 1.69903 2.74224 1.68771 3.35377L1.69479 3.5096Z" fill="url(#paint0_linear_435_759)"/>
                                    <defs>
                                    <linearGradient id="paint0_linear_435_759" x1="11.6389" y1="13.9499" x2="7.18938" y2="1.88497" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.folderView', ['slug' => $slug]) )['color1'] ?? $color1) . '"/>
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.folderView', ['slug' => $slug]) )['color2'] ?? $color2) . '"/>
                                    </linearGradient>
                                    </defs>
                                </svg>  
                            </span>',
                'access' => true,
                'route' => route('user.folderView', ['slug' => 'drive-' . auth()->user()->id ]),
                'menu' => activeMenu(route('user.folderView', ['slug' => $slug]) ),
                'type' => 'common'
            ],
            [
                'id' => 'account',
                'category' => 'System Management',
                'name' => __('Account'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg class="category-svg w-4 h-4" width="18" height="18" viewBox="0 0 18 18" fill="none"
                                xmlns="http://www.w3.org/2000/svg">
                                <path
                                    d="M8.99984 0.666626C4.39984 0.666626 0.666504 4.39996 0.666504 8.99996C0.666504 13.6 4.39984 17.3333 8.99984 17.3333C13.5998 17.3333 17.3332 13.6 17.3332 8.99996C17.3332 4.39996 13.5998 0.666626 8.99984 0.666626ZM8.99984 3.16663C10.3832 3.16663 11.4998 4.28329 11.4998 5.66663C11.4998 7.04996 10.3832 8.16663 8.99984 8.16663C7.6165 8.16663 6.49984 7.04996 6.49984 5.66663C6.49984 4.28329 7.6165 3.16663 8.99984 3.16663ZM8.99984 15C6.9165 15 5.07484 13.9333 3.99984 12.3166C4.02484 10.6583 7.33317 9.74996 8.99984 9.74996C10.6582 9.74996 13.9748 10.6583 13.9998 12.3166C12.9248 13.9333 11.0832 15 8.99984 15Z"
                                    fill="url(#paint0_linear_460_1019)" />
                                <defs>
                                    <linearGradient id="paint0_linear_460_1019" x1="11.5027" y1="15.2819"
                                        x2="5.71732" y2="2.2263" gradientUnits="userSpaceOnUse">
                                        <stop stop-color="' . (activeMenu(route('user.profile'), route('user.package'), route('user.subscription.history'), route('user.subscription.teamList'), route('user.subscription.smallPlan'), route('user.subscription.teamMemberEdit', ['id' => $id]))['color1'] ?? $color1) . '" />
                                        <stop offset="1" stop-color="' . (activeMenu(route('user.profile'), route('user.package'), route('user.subscription.history'), route('user.subscription.teamList'), route('user.subscription.smallPlan'), route('user.subscription.teamMemberEdit', ['id' => $id]))['color2'] ?? $color2) . '" />
                                    </linearGradient>
                                </defs>
                            </svg>
                            </span>',
                'access' => true,
                'route' => route('user.profile'),
                'menu' => activeMenu(route('user.profile'), route('user.package'), route('user.subscription.history'), route('user.subscription.teamList'), route('user.subscription.smallPlan'), route('user.subscription.teamMemberEdit', ['id' => $id])),
                'type' => 'common'
            ]
        ];

        $accessChecks = [
            'template' => [hasAccess('template'), 'user.documents'],
            'code' => [hasAccess('code'), 'user.codeList'],
            'speech_to_text' => [hasAccess('speech_to_text'), 'user.speechLists'],
            'voiceover' => [hasAccess('voiceover'), 'user.voiceoverList'],
            'long_article' => [hasAccess('long_article'), 'user.long_article.index'],
        ];

        $view = '';

        foreach ($accessChecks as $panel => $accessCheck) {
            if ($accessCheck[0] && customerPanelAccess($panel)) {
                $view = route($accessCheck[1]);
                break;
            }
        }

        $history = [
            [
                'id' => 'history',
                'category' => 'System Management',
                'name' => __('History'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg class="category-svg" width="18" height="18" viewBox="0 0 18 18" fill="none"
                                    xmlns="http://www.w3.org/2000/svg">
                                    <path
                                        d="M17.75 9C17.75 11.3206 16.8281 13.5462 15.1872 15.1872C13.5462 16.8281 11.3206 17.75 9 17.75C6.67936 17.75 4.45376 16.8281 2.81282 15.1872C1.17187 13.5462 0.25 11.3206 0.25 9C0.25 8.83424 0.315848 8.67527 0.433058 8.55806C0.550268 8.44085 0.70924 8.375 0.875 8.375C1.04076 8.375 1.19973 8.44085 1.31694 8.55806C1.43415 8.67527 1.5 8.83424 1.5 9C1.4975 10.6738 2.05413 12.3005 3.08154 13.6218C4.10895 14.9432 5.54827 15.8835 7.17105 16.2936C8.79383 16.7036 10.5071 16.5599 12.0389 15.8853C13.5707 15.2106 14.8332 14.0436 15.6262 12.5696C16.4191 11.0955 16.697 9.39886 16.4157 7.74888C16.1344 6.0989 15.31 4.59015 14.0734 3.4621C12.8369 2.33405 11.259 1.65134 9.59018 1.52233C7.92137 1.39332 6.25728 1.82541 4.86204 2.75H5.25C5.41576 2.75 5.57473 2.81585 5.69194 2.93306C5.80915 3.05027 5.875 3.20924 5.875 3.375C5.875 3.54076 5.80915 3.69974 5.69194 3.81695C5.57473 3.93416 5.41576 4 5.25 4H3.375C3.29291 4.00005 3.21162 3.98392 3.13577 3.95252C3.05992 3.92113 2.991 3.8751 2.93295 3.81705C2.87491 3.759 2.82887 3.69009 2.79748 3.61424C2.76609 3.53839 2.74995 3.45709 2.75 3.375V1.5C2.75 1.33424 2.81585 1.17527 2.93306 1.05806C3.05027 0.940853 3.20924 0.875005 3.375 0.875005C3.54076 0.875005 3.69973 0.940853 3.81694 1.05806C3.93415 1.17527 4 1.33424 4 1.5V1.8212C5.31174 0.90694 6.84896 0.369672 8.44463 0.267782C10.0403 0.165893 11.6334 0.50328 13.0507 1.24328C14.4681 1.98328 15.6556 3.09759 16.484 4.46512C17.3125 5.83265 17.7504 7.40109 17.75 9ZM14.625 9C14.625 10.1125 14.2951 11.2001 13.677 12.1251C13.0589 13.0501 12.1804 13.7711 11.1526 14.1968C10.1248 14.6226 8.99376 14.734 7.90262 14.5169C6.81147 14.2999 5.80919 13.7641 5.02252 12.9775C4.23585 12.1908 3.70012 11.1885 3.48308 10.0974C3.26604 9.00624 3.37743 7.87524 3.80318 6.84741C4.22892 5.81958 4.94989 4.94107 5.87492 4.32299C6.79994 3.7049 7.88748 3.375 9 3.375C10.4913 3.37663 11.9211 3.96979 12.9757 5.02433C14.0302 6.07887 14.6234 7.50866 14.625 9ZM11.2217 9.73L9.625 8.66553V5.875C9.625 5.70924 9.55915 5.55027 9.44194 5.43306C9.32473 5.31585 9.16576 5.25 9 5.25C8.83424 5.25 8.67527 5.31585 8.55806 5.43306C8.44085 5.55027 8.375 5.70924 8.375 5.875V9C8.37502 9.10289 8.40043 9.20418 8.44898 9.29489C8.49753 9.3856 8.56772 9.46292 8.65332 9.52L10.5283 10.77C10.6662 10.86 10.8341 10.892 10.9955 10.8589C11.1568 10.8258 11.2986 10.7304 11.39 10.5933C11.4813 10.4563 11.5149 10.2887 11.4834 10.1271C11.4519 9.96539 11.3578 9.8227 11.2217 9.73Z"
                                        fill="url(#paint0_linear_3321_1869)" />
                                    <defs>
                                        <linearGradient id="paint0_linear_3321_1869" x1="11.628" y1="15.5961"
                                            x2="5.55335" y2="1.88766" gradientUnits="userSpaceOnUse">
                                            <stop stop-color="' . (activeMenu(route('user.documents'), route('user.favouriteDocuments'), route('user.editContent', ['slug' => $slug]), route('user.codeList'), route('user.codeView', ['slug' => $slug]), route('user.voiceoverList'), route('user.voiceoverView', ['id' => techEncrypt($id)]), route('user.speechLists'), route('user.editSpeech', ['id' => $id]), route('user.long_article.index'), route('user.long_article.edit', $id))['color1'] ?? $color1) . '" />
                                            <stop offset="1" stop-color="' . (activeMenu(route('user.documents'), route('user.favouriteDocuments'), route('user.editContent', ['slug' => $slug]), route('user.codeList'), route('user.codeView', ['slug' => $slug]), route('user.voiceoverList'), route('user.voiceoverView', ['id' => techEncrypt($id)]), route('user.speechLists'), route('user.editSpeech', ['id' => $id]), route('user.long_article.index'), route('user.long_article.edit', $id))['color2'] ?? $color2 ) . '" />
                                        </linearGradient>
                                    </defs>
                                </svg>
                            </span>',
                'access' => !empty($view) && (customerPanelAccess('template') || customerPanelAccess('voiceover') || customerPanelAccess('code') || customerPanelAccess('speech_to_text') || customerPanelAccess('long_article')),
                'route' => $view,
                'menu' => activeMenu(route('user.documents'), route('user.favouriteDocuments'), route('user.editContent', ['slug' => $slug]), route('user.codeList'), route('user.codeView', ['slug' => $slug]), route('user.voiceoverList'), route('user.voiceoverView', ['id' => techEncrypt($id)]), route('user.speechLists'), route('user.editSpeech', ['id' => $id]), route('user.long_article.index'), route('user.long_article.edit', $id)),
                'type' => 'history'
            ],
            [
                'id' => 'gallery',
                'category' => 'System Management',
                'name' => __('Gallery'),
                'description' => __('Generate stunning images using artificial intelligence.'),
                'icon' => '<span class="w-5 h-5">
                                <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <g clip-path="url(#clip0_12240_6893)">
                                    <path d="M15.0703 1.5C15.0703 1.36739 15.0176 1.24021 14.9239 1.14645C14.8301 1.05268 14.7029 1 14.5703 1H3.57031C3.4377 1 3.31053 1.05268 3.21676 1.14645C3.12299 1.24021 3.07031 1.36739 3.07031 1.5V2H15.0703V1.5Z" fill="url(#paint0_linear_12240_6893)"/>
                                    <path d="M16.0586 3.5C16.0586 3.36739 16.0059 3.24021 15.9121 3.14645C15.8184 3.05268 15.6912 3 15.5586 3H2.55859C2.42599 3 2.29881 3.05268 2.20504 3.14645C2.11127 3.24021 2.05859 3.36739 2.05859 3.5V4H16.0586V3.5Z" fill="url(#paint1_linear_12240_6893)"/>
                                    <path d="M16.06 5H1.94C1.6907 5 1.4516 5.09904 1.27532 5.27532C1.09904 5.4516 1 5.6907 1 5.94V15.06C1 15.3093 1.09904 15.5484 1.27532 15.7247C1.4516 15.901 1.6907 16 1.94 16H16.06C16.3093 16 16.5484 15.901 16.7247 15.7247C16.901 15.5484 17 15.3093 17 15.06V5.94C17 5.6907 16.901 5.4516 16.7247 5.27532C16.5484 5.09904 16.3093 5 16.06 5ZM4.28 6.725C4.57667 6.725 4.86668 6.81297 5.11336 6.9778C5.36003 7.14262 5.55229 7.37689 5.66582 7.65098C5.77935 7.92506 5.80906 8.22666 5.75118 8.51764C5.6933 8.80861 5.55044 9.07588 5.34066 9.28566C5.13088 9.49544 4.86361 9.6383 4.57264 9.69618C4.28166 9.75406 3.98006 9.72435 3.70597 9.61082C3.43189 9.49729 3.19762 9.30503 3.0328 9.05836C2.86797 8.81168 2.78 8.52167 2.78 8.225C2.78 7.82718 2.93804 7.44564 3.21934 7.16434C3.50064 6.88304 3.88218 6.725 4.28 6.725ZM15 14H3L6.73 10.265C6.79649 10.199 6.88635 10.162 6.98 10.162C7.07365 10.162 7.16351 10.199 7.23 10.265L9.07 12.105L11.605 9.5C11.6715 9.43405 11.7613 9.39704 11.855 9.39704C11.9487 9.39704 12.0385 9.43405 12.105 9.5L15 12.395V14Z" fill="url(#paint2_linear_12240_6893)"/>
                                    </g>
                                    <defs>
                                    <linearGradient id="paint0_linear_12240_6893" x1="10.8724" y1="1.87692" x2="10.8378" y2="0.941036" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color1'] ?? $color1) . '"/>
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color2'] ?? $color2) . '"/>
                                    </linearGradient>
                                    <linearGradient id="paint1_linear_12240_6893" x1="11.161" y1="3.87692" x2="11.1314" y2="2.9407" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color1'] ?? $color1) . '"/>
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color2'] ?? $color2) . '"/>
                                    </linearGradient>
                                    <linearGradient id="paint2_linear_12240_6893" x1="11.4027" y1="14.6461" x2="8.52888" y2="5.21289" gradientUnits="userSpaceOnUse">
                                    <stop stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color1'] ?? $color1) . '"/>
                                    <stop offset="1" stop-color="' . (activeMenu(route('user.gallery.show'), route('user.image.index'))['color2'] ?? $color2) . '"/>
                                    </linearGradient>
                                    <clipPath id="clip0_12240_6893">
                                    <rect width="18" height="18" fill="white"/>
                                    </clipPath>
                                    </defs>
                                </svg>
                                    
                            </span>',
                'access' => (hasAccess('image') || hasAccess('video') ) &&  ( customerPanelAccess('image') || customerPanelAccess('video')),
                'route' => route('user.gallery.show'),
                'menu' => activeMenu(route('user.gallery.show'), route('user.image.index')),
                'type' => 'history'
            ],
        ];
        
        return [
            'dashboard' => $dashboard,
            'features' => array_merge($features,  $history, $commonFeatures),
            'categories' => $categories,
            'category_order' => $categoryOrder,
        ];
    }
}