<?php

namespace Modules\OpenAI\AiProviders\OpenAi\Resources;

use App\Rules\CheckValidFile;

class VideoDataProcessor
{
    private array $data = [];

    public function __construct(array $aiOptions = [])
    {
        $this->data = $aiOptions;
    }

    public function rules(): array
    {
        return [
            'service' => [
                'sora-2' => [
                    'image-to-video' => [
                        'service' => false,
                        'prompt' => true,
                        'size' => true,
                        'duration' => true,
                    ],
                ],
                'sora-2-pro' => [
                    'image-to-video' => [
                        'service' => false,
                        'prompt' => true,
                        'size' => true,
                        'duration' => true,
                    ],
                ],
            ],
            'duration' => [
                'sora-2' => [4, 8, 12],
                'sora-2-pro' => [4, 8, 12],
            ],
            'size' => [
                'sora-2' => ['720x1280', '1280x720'],
                // Pro supports the extended resolutions
                'sora-2-pro' => ['720x1280', '1280x720', '1024x1792', '1792x1024'],
            ],
        ];
    }

    public function videoOptions(): array
    {
        return [
            [
                'type' => 'checkbox',
                'label' => 'Provider State',
                'name' => 'status',
                'value' => '',
                'visibility' => true,
            ],
            [
                'type' => 'text',
                'label' => 'Provider',
                'name' => 'provider',
                'value' => 'OpenAi',
                'visibility' => false,
            ],
            [
                'type' => 'dropdown',
                'label' => 'Service',
                'name' => 'service',
                'value' => [
                    'image-to-video',
                ],
                'visibility' => true,
                'admin_visibility' => true,
                'default_value' => 'image-to-video',
                'required' => true,
            ],
            [
                'type' => 'dropdown',
                'label' => 'Models',
                'name' => 'model',
                'value' => [
                    'sora-2',
                    'sora-2-pro',
                ],
                'visibility' => true,
                'required' => true,
                'default_value' => 'sora-2',
            ],
            [
                'type' => 'dropdown',
                'label' => 'Duration',
                'name' => 'duration',
                'value' => [
                    4,
                    8,
                    12,
                ],
                'visibility' => true,
                'required' => true,
            ],
            [
                'type' => 'dropdown',
                'label' => 'Size',
                'name' => 'size',
                'value' => [
                    '720x1280',
                    '1280x720',
                    '1024x1792',
                    '1792x1024',
                ],
                'default_value' => '720x1280',
                'visibility' => true,
                'required' => true,
            ],
            [
                'type' => 'file',
                'label' => 'File',
                'name' => 'file',
                'value' => '',
                'visibility' => true,
                'restrictions' => [
                    'formats' => ['jpeg', 'png'],
                ],
            ]
        ];
    }

    public function generateVideo(): array
    {
        $options = data_get($this->data, 'options', []);
        $model = data_get($options, 'model', 'sora-2');
        $size = data_get($options, 'size', '720x1280');
        $seconds = data_get($options, 'duration') ?? data_get($options, 'seconds');

        $payload = [
            'model' => $model,
            'prompt' => $this->prompt(),
            'size' => $size,
        ];

        // Keep file object for multipart upload
        $file = data_get($this->data, 'options.file');
        if ($file) {
            $payload['input_reference'] = $file;
        }

        if (!is_null($seconds)) {
            $payload['seconds'] = (string) $seconds;
        }

        // Remove nulls; keep prompt/image even if empty string to let validation surface issues.
        return array_filter($payload, fn ($v) => !is_null($v));
    }

    public function videoValidationRules(): array
    {
        $validationRules = [
            'provider' => 'required',
            'prompt' => 'required',
            'options.model' => 'required',
            'options.file' => ['required', new CheckValidFile(getFileExtensions(3))],
        ];

        $validationMessage = [
            'provider.required' => __('Please select a provider.'),
            'prompt.required' => __('Please enter a prompt.'),
            'options.model.required' => __('Please select a model.'),
            'options.file.required' => __('Please select a file.'),
        ];

        return [
            $validationRules,
            $validationMessage,
        ];
    }

    public function findProviderData(array $data, string $searchKey, bool $returnKey = true, array $options = []): ?string
    {
        foreach ($data as $key => $values) {
            if (array_key_exists($searchKey, $values)) {
                return $returnKey ? $key : $values[$searchKey];
            }
        }

        return null;
    }

    public function prepareFile(string $name): ?string
    {
        $uploadedFile = $this->data['options'][$name] ?? null;

        if (!is_null($uploadedFile)) {
            $fileType = $uploadedFile->getMimeType();
            $fileData = base64_encode(file_get_contents($uploadedFile->getRealPath()));
            return 'data:' . $fileType . ';base64,' . $fileData;
        }

        return null;
    }

    public function prompt(): string
    {
        return filteringBadWords("Generate a video based on this concept: `" . $this->data['prompt'] . "`.");
    }
}

